//////////////////////////////////////////////////////////////////////////////
//
// Firework
// Copyright 2004 by Thierry Tremblay
//
// Firework Sample for PocketHAL
// For more info: http://www.droneship.com
//
// Adapted from the Fire Demo at http://www.xdr.com/dash/fire.html
//
//////////////////////////////////////////////////////////////////////////////

#include "firework.h"
#include "../common/utility.h"

#if defined(PHAL_PLATFORM_SYMBIAN)
#include <stdlib.h>
#endif



//////////////////////////////////////////////////////////////////////////////
//
// Declare the program's entry point
//
//////////////////////////////////////////////////////////////////////////////

PHAL_DECLARE_ENTRY_POINT( Firework );




//////////////////////////////////////////////////////////////////////////////
//
// Constants
//
//////////////////////////////////////////////////////////////////////////////

const int MAX_BLOBS = 512;
const int RATE      = 1;
const int SMALLSIZE = 3;
const int BIGSIZE   = 6;
const int BLOBFRAC  = 6;
const int BLOBGRAVITY = 5;
const int THRESHOLD = 20;


#define ABS(x) ((x)<0 ? -(x) : (x))

const char sizes[] = { 2,3,4,5,8,5,4,3 };

const char sqrttab[] =
{
    0,1,1,1,2,2,2,2,2,3,3,3,3,3,3,3,
    4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,
    5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,
    6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
    8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,
    9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,
    10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,
    11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
    12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
    12,12,12,12,12,12,12,12,12,13,13,13,13,13,13,13,
    13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
    13,13,13,13,14,14,14,14,14,14,14,14,14,14,14,14,
    14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,
    14,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,
    15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15
};



//////////////////////////////////////////////////////////////////////////////
//
// Build a RGB565 value
//
//////////////////////////////////////////////////////////////////////////////

inline int RGB565( int red, int green, int blue )
{
    red   <<= 2;
    green <<= 2;
    blue  <<= 2;

    return ((red << 8) & (0x1F<<11)) | ((green << 3) & (0x3F<<5)) | (blue >> 3);
}



//////////////////////////////////////////////////////////////////////////////
//
// Firework
//
//////////////////////////////////////////////////////////////////////////////

Firework::Firework()
:   m_buffer1(0),
    m_buffer2(0),
    m_map1(0),
    m_map2(0),
    m_blobs(0)
{
    m_config.m_appName     = TEXT("Firework");
    m_config.m_orientation = ORIENTATION_NORMAL;
}



Firework::~Firework()
{
    delete [] m_buffer1;
    delete [] m_buffer2;
    delete [] m_map1;
    delete [] m_map2;
    delete [] m_blobs;
}



bool Firework::OnInitialize()
{
    if (!Game::OnInitialize())
        return false;
 
    m_width  = GetDisplay()->GetParameters().m_width;
    m_height = GetDisplay()->GetParameters().m_height;
    
    // Buffers
    int size = m_width * m_height;

    m_buffer1 = new uint8_t[size];
    m_buffer2 = new uint8_t[size];

    memset( m_buffer1, 0, size );
    memset( m_buffer2, 0, size );


    // Map
    m_map1 = new uint8_t[8192];
    m_map2 = new uint8_t[8192];

    int i;

    for (i = 0; i != 8192; ++i)
    {
        int j = i / 9;
        m_map1[i] = j < 256 ? (j >= RATE ? j - RATE : 0 ) : 255;
    }

    for (i = 0; i != 8192; ++i)
    {
        m_map2[i] = (i>>3) < 255 ? (i>>3) : 255;
    }


    // Blobs
    m_blobs = new Blob[MAX_BLOBS];

    m_freeBlobs = m_activeBlobs = 0;
    for (i = 0; i != MAX_BLOBS; ++i)
    {
        m_blobs[i].next = m_freeBlobs;
        m_freeBlobs = m_blobs + i;
    }


    // Palette
    SetupPalette();

    
    // Initialize the random seed
    srand( PHAL::GetClock() );


    // Initialize fire variables
    m_flash   = 0;
    m_explode = 0;
    m_now     = 0;

    return true;
}



bool Firework::OnGameLoop()
{
    Display* display = GetDisplay();

    if (display->BeginScene())
    {
        // Render one frame
        Draw( display->GetBackBuffer() );
        DrawText( display->GetBackBuffer(), 0, 0, GetFPSString() );
        display->Swap();

        // Animate (game logic)
        Animate();
    }

    return Game::OnGameLoop();
}



void Firework::SetupPalette()
{
    for (int i = 0; i != 256; ++i)
    {
        if (i < 64) m_palette[i] = RGB565( 0, 0, 0 );
        else if (i < 128) m_palette[i] = RGB565( i-64, 0, 0 );
        else if (i < 192) m_palette[i] = RGB565( 63, i-128, 0 );
        else m_palette[i] = RGB565( 63, 63, i-192 );
    }
}





void Firework::Animate()
{
    // Animate the fire
    ++m_now;

    if (m_flash)
    {
        --m_flash;
    }
    else
    {
        if ( (m_explode > 96 && m_explode < 160) && !(rand()&511) ) //todo: || (buttonstate&8)
        {
            m_flash = 60;
        }
    }

    m_explode = (m_now >> 4) + 1;
    if (m_explode == 320) m_now = 0;
    if (m_explode > 256) m_explode = 256;

    if (!(rand() & 31))
        AddBlob();

    MoveBlobs();
    PutBlobs();

    DoFire( m_buffer2, m_buffer1, m_width, m_flash ? m_map2 : m_map1 );


    // Swap the buffers
    uint8_t* temp = m_buffer1;
    m_buffer1 = m_buffer2;
    m_buffer2 = temp;
}




void Firework::Draw( PHAL::Surface* backbuffer ) const
{
    const uint8_t* src = m_buffer2;

    for (int y = 0; y != m_height; ++y)
    {
        uint16_t* dest = backbuffer->GetPixels( 0, y );

        for (int x = 0; x != m_width; ++x)
        {
            *dest++ = m_palette[*src++];
        }
    }
}




void Firework::AddBlob()
{
    if (!m_freeBlobs)
        return;
    
    Blob* blob  = m_freeBlobs;
    m_freeBlobs = m_freeBlobs->next;

    blob->x     = (32 + (rand() % (m_width - 64)))<<BLOBFRAC;
	blob->y     = 32<<BLOBFRAC;
	blob->dx    = (rand() & 0xFF) - 128;
	blob->dy    = (rand() % 200)  + 160;
    blob->life  = (rand() & 511)  + 256;
	blob->size  = BIGSIZE;
	
	blob->next    = m_activeBlobs;
    m_activeBlobs = blob;
}



void Firework::MoveBlobs()
{
    Blob **lastBlob, *blob;
    int x,y;
    
    lastBlob = &m_activeBlobs;
    
    while ((blob = *lastBlob) != 0)
    {
        x = blob->x >> BLOBFRAC;
        y = blob->y >> BLOBFRAC;
    
        --blob->life;

        if(!blob->life || y<0 || x<10 || x>m_width-10)
        {
            *lastBlob   = blob->next;
            blob->next  = m_freeBlobs;
            m_freeBlobs = blob;
            continue;
        }
        
        blob->x  += blob->dx;
        blob->y  += blob->dy;
        blob->dy -= BLOBGRAVITY;
        lastBlob = &blob->next;
    }
}



void Firework::PutBlobs()
{
    Blob *ablob, *ablob2, *temp;
    int x,y,dy;
    int i,size;
    long x2,y2,vel;
    
    ablob= m_activeBlobs;
    m_activeBlobs=0;
    while(ablob)
    {
        dy=ablob->dy;
        if(ablob->size!=SMALLSIZE && (dy>-THRESHOLD && dy<THRESHOLD && !(rand()&7) || (rand()&127)==63))
        {
            i=m_explode;
            while(i-- && m_freeBlobs)
            {
                ablob2=m_freeBlobs;
                m_freeBlobs=m_freeBlobs->next;
                ablob2->x=ablob->x;
                ablob2->y=ablob->y;
                for(;;)
                {
                    x2=(rand()&511)-256;
                    y2=(rand()&511)-256;
                    vel=x2*x2+y2*y2;
                    if(vel>0x3000 && vel<0x10000L) break;
                }
                ablob2->dx=ablob->dx+x2;
                ablob2->dy=ablob->dy+y2;
                ablob2->life=16+(rand()&31);
                ablob2->size=SMALLSIZE;
                ablob2->next=m_activeBlobs;
                m_activeBlobs=ablob2;
                ablob->life=1;
            }			
        }
        x=ablob->x>>BLOBFRAC;
        y=ablob->y>>BLOBFRAC;
        size=ablob->size;
        
        if(size==BIGSIZE && ablob->dy>0 && ablob->dy<200)
            size=sizes[ablob->life&7];
        
        if(x>10 && x<m_width-10 && y>10 && y<m_height-10)
            Disk(x,m_height-1-y,size);
        
        temp=ablob;
        ablob=ablob->next;
        temp->next=m_activeBlobs;
        m_activeBlobs=temp;
    }
}



void Firework::Disk( int x, int y, int rad )
{
    int w;
    uint8_t* p;
    int i,j,k,aj;
    int rr = rad * rad;
    
    for(j=-rad;j<=rad;j++)
    {
        w=sqrttab[rr-j*j];
        aj=ABS(j)<<2;
        if(w)
        {
            p = m_buffer2 + (y+j)*m_width + x-w;

            k=w+w+1;
            i=-w;
            while(k--) {*p++=255-(ABS(i)<<2)-aj;i++;}
        }
    }
}




void Firework::DoFire( const uint8_t* p1, uint8_t* p2, int pitch, const uint8_t* map )
{
    int x,y;
    const uint8_t* p3;
    uint8_t* p4;

	for(y=2;y<m_height;y++)
	{
		for(x=0;x<m_width;x++)
		{
			p3 = p1 + y*m_width + x;
			p4 = p2 + y*pitch + x;
			*p4=map[*p3+p3[-m_width]+p3[-m_width-1]+p3[-m_width+1]+p3[-1]+p3[1]+p3[-m_width-m_width-1]+p3[-m_width-m_width]+p3[-m_width-m_width+1]];
		}
	}
}
