/* Copyright 2010 Palm, Inc.  All rights reserved. */

/*jslint browser: true, devel: true, laxbreak: true, white: false */
/*global $L, browser, Calendar, Class, DayAssistant, Event, Foundations */
/*global getAppAssistant, Hash, JumptoDialogAssistant, Mojo, WeekAssistant */

var MONTH_DAY_ID_FORMAT = "MMMM dd yyyy";	// IMPORTANT: DO NOT LOCALIZE!

var MonthAssistant = Class.create({

	reminderMenuItemId: 3,

	initialize: function() {
		this.appMenuModel = { visible:true, 
					label:$L('Calendar'), 
					items: [
						Mojo.Menu.editItem,
						{label:$L('Sync Now'), command:'sync', id: 0},
						{label:$L('Show today'), command:'today', id: 1},
						{label:$L('Jump to...'), command:'jumpto', id: 2},
						{label:$L('Missed reminders...'), command:'reminders', id: 3},
						{label:$L('Preferences & Accounts'), command:Mojo.Menu.prefsCmd, checkEnabled: true},
						{label:$L('Help'), command:Mojo.Menu.helpCmd, disabled:false}
					]
				};

		this.app				= getAppAssistant();
		this.calendarsManager	= this.app.getCalendarsManager();
		this.prefsManager		= this.app.getPrefsManager();
		this.eventManager		= new Calendar.EventManager();

		// Setup callbacks. Bind "this" references only once, at initialization:
		this.databaseChanged		=
		this.loadBusyTimes			= this.loadBusyTimes		.bind (this);
		this.loadBusyTimesBatch		= this.loadBusyTimesBatch	.bind (this);
		this.renderBusyTimes		= this.renderBusyTimes		.bind (this);
		this.locale					= Mojo.Locale.getCurrentLocale();
///*DEBUG:*/	this.timing = { launch: 0, mgr: 0 };
	},

	loadBusyTimes: function (delay) {
///*DEBUG:*/	this.timing.mgr = -new Date();

		if (!isNaN (delay)) {
			clearTimeout (this.loadBusyTimes.thread);
			this.loadBusyTimes.thread = setTimeout (this.loadBusyTimes, delay);
			return;
		} else {  delete this.loadBusyTimes.thread; }

		var start	= new Date (this.firstDay).clearTime()
		,	end		= new Date (this.lastDay);

		end.set ({ hour: 23, minute: 59, second: 59, millisecond: 0 });

		var range =
		{	start		: start	.addWeeks(6)	// Limit range to between the first
		,	end			: end	.addWeeks(-6)	// and last weeks displayed instead of full 18-weeks. 
		,	tzId		: this.app.getTimezoneName()
		,	excludeList	: this.calendarsManager.getExcludeFromAllList()		
		};

		var calendarId = this.calendarsManager.getCurrentCal();
		if (calendarId != "all") {
			range.calendarId = calendarId;
		}

		this.loadBusyTimesBatch (range);
	},

	loadBusyTimesBatch: function (range, end) {
		/* Loads busy times in "asynchronous" chunks to minimize delay in view rendering.*/

		end		= (end		&& (this.loadBusyTimesBatch.end		= end	)) || this.loadBusyTimesBatch.end;
		range	= (range	&& (this.loadBusyTimesBatch.range	= range	)) || this.loadBusyTimesBatch.range;
																									//console.log ("\n\n\nloadBusyTimesBatch:\n\targs: "+arguments.length+" - "+JSON.stringify(arguments)+"\n\tend: "+end+", range: "+JSON.stringify(range)+"\n\n\n");
		if (!range || (end && end <= range.end)) {							// All busy-time batches processed so
			delete this.loadBusyTimesBatch.range;							// clear the batch processor's range
			delete this.loadBusyTimesBatch.start;							// clear the batch processor's start
			delete this.loadBusyTimesBatch.end;								// clear the batch processor's end
																									//console.log ("\n\n\nloadBusyTimesBatch: DONE\n\n\n");
			return;															// then exit.
		}

		if (end) {
			range.start = new Date(range.end).addDays(-1).getTime();			// Batch continuation so move range start ahead 3 weeks.
																				// monthview renders one day less than the range requested, so we'll add
																				// a day to our end date.  We need to back up a day to get the right start if this
																				// is a continuation. 

		} else {															// First busy-time batch so
			this.loadBusyTimesBatch.start	= range.start;					// cache the range start and
			this.loadBusyTimesBatch.end		= range.end.getTime();			// end dates and
			range.start						= range.start.getTime();		// set the range start timestamp.
		}

		
		range.end = this.loadBusyTimesBatch.start.addWeeks(3).addDays(1).getTime();	// Set range end 3 weeks after start.
																					// Add one extra day because monthview renders one day less than the range requested
																					//i.e., string length of 35 = render 34 days of info

		if (range.end > this.loadBusyTimesBatch.end) {						// If batch's end is out of range
			range.end = this.loadBusyTimesBatch.end;						// reset to range's end.
		}

		var callback = this.onLoadBusyTimesBatch.bind (this, range);
		this.eventManager.getBusyDays (range, callback);					// Request next batch of busy times.
	},

	onLoadBusyTimesBatch: function (range, busyTimeInfo) {											//console.log ("\n\n\nonLoadBusyTimesBatch:\n\targs: "+JSON.stringify(arguments)+"\n\n\n");
		//	Possible Improvements:
		//	- Throttle to avoid flashing.
		//	- Aggregate as many batches as possible within a predetermined interval.
		//	- Cache results to avoid EventManager recalculation if time range is revisited.
		this.renderBusyTimes (busyTimeInfo);			// Immediately render available busy-times.

		if (range != this.loadBusyTimesBatch.range) {												//console.log ("\n\n\nonLoadBusyTimesBatch: new batch abort:\n\targs: range:"+JSON.stringify(range)+"\n\tnew.range:"+JSON.stringify( this.loadBusyTimesBatch.range)+"\n\n\n");
			return;										// New range so stop current batch.
		}
		setTimeout (this.loadBusyTimesBatch, 15.625);	// Delay next request to allow GUI update and interaction.
	},

	buildViewHeader: function() 
	{
		//Mojo.Log.info("month-assistant: buildViewHeader");
		this.controller.get('mv_header_container1').update(Mojo.View.render({object: { viewPrefix: 'mv' }, template: 'shared/calendar_view_header'}));
		this.mvTitle = this.controller.get('mv_title');
	},
	
	buildMonthTitle: function() 
	{
		var formatObj = {};
        // formatString = "MMM yyyy";
        formatObj.date = "medium";
        formatObj.dateComponents = "my";

		//this.mvTitle.innerText = Mojo.Format.formatDate(this.monthDate, $L("MMM yyyy")); // Localize this date format string
		this.mvTitle.innerText = Mojo.Format.formatDate(this.monthDate, formatObj); // Localize this date format string
	},

	buildDayHeader: function() {
		//Mojo.Log.info("month-assistant: buildDayHeader");
		var genDayHeader = '';
		
		// Set up the array of day names based on the start of the week
		var date = new Date();
		var dayValue = 0;
		if (this.savedPrefs) {
			dayValue = this.savedPrefs.startOfWeek - 1;
		}
			
		var days = [];
		
		date.moveToDayOfWeek(dayValue);
		
		for (var i = 0; i < 7; i++) {
			var dayName = Mojo.Format.formatDate(date, "E");
			days.push(dayName[0]);
			// Generate the html for the row containing the day header
			var dayOfWeek = date.getDay();
			if (dayOfWeek === 0 || dayOfWeek == 6) {
				// Weekend
				genDayHeader += '<div class="day weekend"> ' + days[i] + ' </div>';
			} else {
				// Weekday
				genDayHeader += '<div class="day"> ' + days[i] + ' </div>';
			}
			
			date.addDays(1);
		}	
		
		this.controller.get('mv_day_labels').update(genDayHeader);
	},
	
	_days: [ 'one', 'two', 'three', 'four', 'five', 'six', 'seven' ],
	
	_months: null,
	
	// Create a table for the empty months.  NOTE: total weeks is always 18 (126 days).
	_createEmptyMonthTable: function()
	{
		var t = document.createElement('table');
		t.className = 'months';
		for (var w = 0; w < 18; w++)
		{
			var r = t.insertRow(w);
			r.className = 'week';
			for (var d = 0; d < 7; d++)
			{
				var c = r.insertCell(d);
				c.className = 'day';
				c.setAttribute('x-mojo-tap-highlight','momentary');
				var i = c.appendChild(document.createElement('div'));
				i.className = 'monthview-free-time';
				i = c.appendChild(document.createElement('div'));
				i.className = 'monthview-free-time';
				i = c.appendChild(document.createElement('div'));
				i.className = 'monthview-free-time';
				i = c.appendChild(document.createElement('div'));
				i.className = 'monthview-day-numeral';
			}
		}
		
		return t;
	},
	
	//Converts a character [0-9|A-Z|a-z|+-] to a number in range 0 - 63.
	//The number is actually a bitflag used to represent busytime.
	_token2state: function(token)
	{
		//48 = ASCII character 0
		//57 = ASCII character 9
		if (token >= 48 && token <= 57)
		{
			//return 0-9 for 0-9
			return token - 48;
		}
		//65 = ASCII character A
		//90 = ASCII character Z
		else if (token >= 65 && token <= 90)
		{
			//return 10-35 for A-Z
			return token - 55;
		}
		//97 = ASCII character a
		//122 = ASCII character z
		else if (token >= 97 && token <= 122)
		{
			//return 36-61 for a-z
			return token - 61;
		}
		else if (token == '+')
		{
			return 62;
		}
		else
		{
			return 63;
		}
	},
	
	// A dummy week - used to filling out new week views
	_dummyWeek: null,
	
	_createDummyWeek: function()
	{
		var item = { className: 'monthview-free-time' };
		item.nextSibling = item;
		var day = { firstChild: item }; 
		day.nextSibling = day;
		day.previousSibling = day;
		this._dummyWeek = { firstChild: day, lastChild: day };
	},
	
	// Days in each of 12 months (Feb is fixed up by hand as necessary)
	// We try to avoid the DateJS stuff because it's indescribably slow.
	_monthDays: [ 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ],
	
	// Render a week forwards - used when we're scrolling forwards in time
	_renderWeekForward: function(week, ridx, oldweek, mday)
	{
		ridx *= 7;
		var day = week.firstChild;
		var oday = oldweek.firstChild;
		for (var c = 0; c < 7; c++) 
		{
			var child = day.firstChild;
			var ochild = oday.firstChild;
			var thisday = mday.day;
			
			// Don't set CSS if the value is the same.  You'd think this wouldn't do anything
			// but WebKit considers all attribute changes to have potential side effects - so this
			// does much damage to performance and is best avoided
			for (var i = 0; i < 3; i++)
			{
				var cn = ochild.className;
				if (child.className != cn) 
				{
					child.className = cn;
				}
				child = child.nextSibling;
				ochild = ochild.nextSibling;
			}
			if(this.locale=="zh_cn"){
				var luna=iCan_LunarDetail(mday.year,mday.month,mday.day);
				var html = "<span class='inner_day'>"+thisday+"</span><span class='inner_cday'>"+(luna.solarTerm?luna.solarTerm:luna.day==1?(luna.isLeap?$L("Leap"):"")+luna.cMonth:luna.cDay)+"</span>";
				child.innerHTML = html;
			}else{
				child.innerHTML = thisday;
			}
			
			var cls = 'day ' + (thisday <= 7 ? this._days[thisday - 1] : '');
			if (ridx + c - (thisday - 1) == this.thisMonthStartIndex) 
			{
				cls += ' selected-month' + (ridx + c == this.thisDayIndex ? ' today' : '');
				if (mday.dayOfWeek === 0 || mday.dayOfWeek == 6) {
					cls += ' weekend';
				}
			}
			if (day.className != cls) 
			{
				day.className = cls;
			}
			
			// Move the day forwards, allowing for month endings
			if (thisday < mday.limit)
			{
				mday.day++;
			}
			else
			{
				if (mday.month < 11)
				{
					mday.month++;
				}
				else
				{
					mday.month = 0;
					mday.year++;
				}
				mday.limit = this._monthDays[mday.month];
				mday.day = 1;
			}
			
			if (mday.dayOfWeek == 6) {
				mday.dayOfWeek = 0;
			} else {
				mday.dayOfWeek++;
			}
			
			day = day.nextSibling;
			oday = oday.nextSibling;
		}
	},
	
	// Render a week backwards - used when we're scrolling backwards in time
	_renderWeekBackward: function(week, ridx, oldweek, mday)
	{
		ridx *= 7;
		var day = week.lastChild;
		var oday = oldweek.lastChild;
		for (var c = 6; c >= 0; c--) 
		{
			var child = day.firstChild;
			var ochild = oday.firstChild;
			var thisday = mday.day;
			
			for (var i = 0; i < 3; i++)
			{
				var cn = ochild.className;
				if (child.className != cn) 
				{
					child.className = cn;
				}
				child = child.nextSibling;
				ochild = ochild.nextSibling;
			}

			if(this.locale=="zh_cn"){
				var luna=iCan_LunarDetail(mday.year,mday.month,mday.day);
				var html = "<span class='inner_day'>"+thisday+"</span><span class='inner_cday'>"+(luna.solarTerm?luna.solarTerm:luna.day==1?(luna.isLeap?$L("Leap"):"")+luna.cMonth:luna.cDay)+"</span>";
				child.innerHTML = html;
			}else{
				child.innerHTML = thisday;
			}
			
			var cls = 'day ' + (thisday <= 7 ? this._days[thisday - 1] : '');
			if (ridx + c - (thisday - 1) == this.thisMonthStartIndex) 
			{
				cls += ' selected-month' + (ridx + c == this.thisDayIndex ? ' today' : '');
				if (mday.dayOfWeek === 0 || mday.dayOfWeek == 6) {
					cls += ' weekend';
				}
			}
			if (day.className != cls) 
			{
				day.className = cls;
			}
			
			if (thisday > 1)
			{
				mday.day--;
			}
			else
			{
				if (mday.month > 0)
				{
					mday.month--;
				}
				else
				{
					mday.month = 11;
					mday.year--;
				}
				mday.limit = this._monthDays[mday.month];
				mday.day = mday.limit;
			}
			
			if (mday.dayOfWeek === 0) {
				mday.dayOfWeek = 6;
			} else {
				mday.dayOfWeek--;
			}
			
			day = day.previousSibling;
			oday = oday.previousSibling;
		}
	},
	
	render: function() {

		var rows = this._months.rows;
		var date = new Date(this.firstDay);
		var r;
		var mday;
		
		// We use our own date calculator because the system DateJS stuff is indescribably slow (increases this methods speed by 66%).
		// Fix the leapyears so we display the correctly
		if (Date.isLeapYear(date.getFullYear() + (date.getMonth() >= 2 ? 1 : 0)))
		{
			this._monthDays[1] = 29;
		}
		else
		{
			this._monthDays[1] = 28;
		}
		
		// Calculate the difference (in weeks) between what we're rendering now, and what we did last time
		var wdiff = ((this.firstDay.getTime() - this.lastFirstDay.getTime()) / 604800000) | 0; // 1000*60*60*24*7 - 1 week in milliseconds
		// If the different is between -17 and +17 (inc) then we copy up the weeks from the last render (so keeping the busy info correct)
		// and avoiding flicker as much as we can.  We fill in any missing weeks with a dummy week (basically all days marked as free).
		if (wdiff > 0 && wdiff < 18) 
		{
			mday = { day: date.getDate(), limit: 0, month: date.getMonth(), year: date.getFullYear() };
			mday.limit = this._monthDays[mday.month];
			mday.dayOfWeek = date.getDay();
		
			for (r = 0; r < 18 - wdiff; r++) 
			{
				this._renderWeekForward(rows[r], r, rows[r + wdiff], mday);
			}
			for (; r < 18; r++) 
			{
				this._renderWeekForward(rows[r], r, this._dummyWeek, mday);
			}
		}
		else if (wdiff < 0 && wdiff > -18)
		{
			date.addWeeks(18).addDays(-1);
			mday = { day: date.getDate(), limit: 0, month: date.getMonth(), year: date.getFullYear() };
			mday.limit = this._monthDays[mday.month];
			mday.dayOfWeek = date.getDay();
		
			for (r = 17; r > -wdiff; r--)
			{
				this._renderWeekBackward(rows[r], r, rows[r + wdiff], mday);
			}
			for (; r >= 0; r--) 
			{
				this._renderWeekBackward(rows[r], r, this._dummyWeek, mday);
			}
		}
		else 
		{
			// When all else fails, we just create a free set of weeks
			mday = { day: date.getDate(), limit: 0, month: date.getMonth(), year: date.getFullYear() };
			mday.limit = this._monthDays[mday.month];
			mday.dayOfWeek = date.getDay();
		
			for (r = 0; r < 18; r++) 
			{
				this._renderWeekForward(rows[r], r, this._dummyWeek, mday);
			}
		}
		
		// Remember the first day for next time
		this.lastFirstDay = new Date(this.firstDay);
	},
	
	//Monthview expects a response that looks like this: 
	//{"date": 1261296000000, 
	//"days": "000100000016000a200610a000000a000600a09020000026R01J7331000403100000310000061000002R0000063000000300000630000001000000100000010"}
	//Each character in the days string is the busy info for that day.
	//Use _token2state to convert the character into a number.
	//The number is a bitflag:
	//00000000
	//       ^-------- Bit 1: morning event
	//      ^--------- Bit 2: midday event
	//     ^---------- Bit 3: evening event
	//    ^----------- Bit 4: morning event is part of current displayed calendar, so use a special color
	//   ^------------ Bit 5: midday event is part of current displayed calendar, so use a special color
	//  ^------------- Bit 6: evening event is part of current displayed calendar, so use a special color
	// ^-------------- Bit 7: not used
	//^--------------- Bit 8: not used
	
	//Example: days[48] = 'R'  
	//token2state('R') = 27
	//binary of 27 = 00011011
	//                      ^-------- morning event: YES
	//                     ^--------- midday event: YES
	//                    ^---------- evening event: NO
	//                   ^----------- morning event is part of current displayed calendar: YES
	//                  ^------------ midday event is part of current displayed calendar: YES
	//                 ^------------- NO evening event: NO
	//                ^-------------- not used
	//               ^--------------- not used
	renderBusyTimes: function (response) {

///*DEBUG:*/	this.timing.mgr	+= +new Date();

		//Mojo.Log.info("*********** response in month %j", response);

		var	calStyle	= ''
		,	currentCal	= this.calendarsManager.getCurrentCal();

		if (currentCal != "all") {
			calStyle = this.calendarsManager.getCalColorStyle (currentCal);
		}

		// Calculate the day display position of the response date:
		var	firstDayStamp	= (new Date (this.firstDay)).set({hour: 0, minute: 0, second: 0, millisecond: 0}).getTime()
		,	startDayIndex	= Math.round ((response.date - firstDayStamp) / 86400000)
		,	weekCell		= this._months.rows [(startDayIndex / 7) | 0]
		,	dayCell			= weekCell && weekCell.cells [startDayIndex % 7]
		,	days			= response.days
		,	daylen			= days.length - 1; // Ignore extra day; inclusive rather than exclusive response.

		for (var child, day, d=0; dayCell && (d < daylen); d++)
		{
			day		= this._token2state (days.charCodeAt (d));
			child	= dayCell.firstChild;

			for (var cls, i = 1; i < 8; i <<= 1)
			{
				//check bits 4-6 for an event on the displayed calendar
				//this assumes that if the higher bits are 1, so is the corresponding lower bit
				if (day & (i << 3)) 
				{
					cls = "monthview-busy-time " + calStyle;
				}
				//check bits 1-3 for an event
				else if (day & i) 
				{
					cls = "monthview-busy-time";
				}
				else 
				{
					cls = "monthview-free-time";
				}
				if (child.className != cls) 
				{
					child.className = cls;
				}
				child = child.nextSibling;
			}
			
			// Advance the dayCell
			if (d % 7 == 6) 
			{
				// Advance to the next week
				weekCell = weekCell.nextSibling;
				dayCell = weekCell ? weekCell.firstChild : null;
			}
			else 
			{
				dayCell = dayCell.nextSibling;
			}
		}
///*DEBUG:*/	console.log ("\n\n\nmonth-view.renderBusyMonthInfo: timing: "+JSON.stringify (this.timing)+"\n\n\n");
	},
	
	positionDays: function() 
	{
		//Mojo.Log.info("month-assistant: positionDays");
		// Given the month and year, build a table of days.  May include
		// days from the previous and next months
		var currentDateTime			= this.app.getCurrentDateTime();
		this.firstOfTheMonth		= (new Date(currentDateTime)).moveToFirstDayOfMonth();
		this.firstOfPreviousMonth	= (new Date(this.firstOfTheMonth)).addMonths (-1);
		this.firstOfNextMonth		= (new Date(this.firstOfTheMonth)).addMonths (1);

		//var startOfWeek = new Date(this.firstOfTheMonth);
		var daysFromPreviousMonth = this.firstOfTheMonth.getDay();

		if (this.savedPrefs) {
			//startOfWeek.moveToDayOfWeek(this.savedPrefs.startOfWeek-1);

			// Figure out which days from the previous month are visible
			daysFromPreviousMonth = this.firstOfTheMonth.getDay() - (this.savedPrefs.startOfWeek - 1);

			if (daysFromPreviousMonth < 0) {
				daysFromPreviousMonth += 7;
			}
		}

		// 6 weeks before the first of the month
		daysFromPreviousMonth += 42;
		
		this.firstDay = new Date(this.firstOfTheMonth);
		this.firstDay.addDays(-daysFromPreviousMonth);
		
		if (!this.lastFirstDay) 
		{
			this.lastFirstDay = this.firstDay;
		}
		
		var totalDays = 126;
		this.lastDay = new Date(this.firstDay);
		this.lastDay.addDays(totalDays);
		
		// Note: 1 day == 1000*60*60*24 ms == 86400000	
		
		// Use firstDay, noon to determine what thisDayIndex is... If we don't set it to noon,
		// due to rounding thisDayIndex may get set incorrectly
		var firstDay = new Date(this.firstDay);	
		var fidx = firstDay.set({hour: 12, minute: 0, second: 0, millisecond: 0}).getTime();
		// Use today, noon to determine what thisDayIndex is...  If we don't set it to noon,
		// due to rounding thisDayIndex may get set incorrectly if todayDate is set too early
		// or late.
		var todayDate = (new Date()).set({hour: 12, minute: 0, second: 0, millisecond: 0}).getTime();
		this.thisDayIndex = Math.round((todayDate - fidx) / 86400000);
		this.thisMonthStartIndex = Math.round((this.firstOfTheMonth.getTime() - fidx) / 86400000);
		this.prevMonthStartIndex = Math.round((this.firstOfPreviousMonth.getTime() - fidx) / 86400000);
		this.nextMonthStartIndex = Math.round((this.firstOfNextMonth.getTime() - fidx) / 86400000);
	},

	buildMonth: function() {
		this.buildMonthTitle();

		if (this.isScrolling) {
			return;
		}
		this.positionDays();
		this.render();
		this.showToday();
		this.resetScrolling (this._months);
		this.loadBusyTimes (1000);
	},
	
	resetScrolling: function (monthsElement) {
		//Mojo.Log.info ("Updating month view's scroller model snap elements.");

		var	weekRows = monthsElement && monthsElement.rows;
		if (!weekRows) {
			Mojo.Log.error ("Failed to update month view's scroller model snap elements.");
			return;
		}

		this.updateScrollerModel (
		{	snapElements:													// Set scroller's snap elements
			{	y:															// for the y-axis (vertical) to
				[	weekRows [((this.prevMonthStartIndex + 21) / 7) | 0]	// 1st day in previous month,
				,	weekRows [((this.thisMonthStartIndex + 21) / 7) | 0]	// 1st day in current month,
				,	weekRows [((this.nextMonthStartIndex + 21) / 7) | 0]	// 1st day in next month
				]
			}
		,	snapIndex: { value:1 }											// Snap to the current month.
		});
	},

	// *** OBSERVER CALLBACKS ***
	currentDateTimeUpdated: function() {
		// This is the main trigger point for Month View rendering called by
		// the app assistant when this.onMove sets the current Date & Time.

		var	curDateTime		=	this.app.getCurrentDateTime()
		,	isNewMonth		=	this.firstOfTheMonth.getMonth()	!= curDateTime.getMonth()
							||	this.firstOfTheMonth.getYear()	!= curDateTime.getYear()
		;	this.monthDate	=	curDateTime;

		// Check to see if we even need to update the view, since the current date could
		// still be represented by the month currently in view
		if (isNewMonth) {
			this.buildMonth();
		}
	},

	currentCalendarUpdated: function() {
		//Mojo.Log.info("month-assistant: currentCalendarUpdated");
		var calMgr	= this.calendarsManager
		,	calendar= calMgr.getCurrentCal();

		if ("all" == calendar) {
			this.controller.get ("mv_current_calendar")	.update			($L("All"));
			this.controller.get ("mv_calendar_source")	.setAttribute	("class", "header-cal-source");
			return;
		}

		// Special case the local calendar to always show Palm since Palm Profile always
		// gets truncated and looks ugly
		if ("Local" == calMgr.getCalSyncSource (calendar)) {
			// Special case the local calendar to always show Palm since Palm Profile always
			// gets truncated and looks ugly
			this.controller.get ("mv_current_calendar").update ($L("Palm"));
		} else {
			this.controller.get ("mv_current_calendar").update (calMgr.getCalAccountName (calendar));
		}

		this.controller.get ("mv_calendar_source").setAttribute ("class", "header-cal-source " + calMgr.getCalColorStyle (calendar));
	},

	calendarSettingsUpdated: function() {
		//Mojo.Log.info('month-assistant: calendarSettingUpdated  %s' , this.calendarsManager.getCurrentCal());

		// If "all" calendars are displayed, no colors are displayed so no need to update:
		if (this.calendarsManager.getCurrentCal() != "all") {
			this.pendingCalendarSettingsUpdate = true;
			this.buildMonth();
		}
	},

	calendarListUpdated: function() {
		// Do nothing
	},

	calendarPrefsUpdated: function() {
		var newPrefs = this.prefsManager.getPrefs();
		
		// If the start of week changes, then we need to update the
		// day headers and the whole month layout
		if (newPrefs.startOfWeek != this.savedPrefs.startOfWeek) {
			//Mojo.Log.info('month-assistant: calendarPrefsUpdated startOfWeek');
			this.savedPrefs.startOfWeek = newPrefs.startOfWeek;
			this.buildDayHeader();
			this.buildMonth();
		}
		
		this.savedPrefs = Foundations.ObjectUtils.clone (newPrefs);
	},
	
	dayChanged: function() {
		//Mojo.Log.info("month-assistant: dayChanged");
		this.buildMonth();
	},

	// ***
	setup: function() {

///*DEBUG:*/	this.timing.launch = -new Date();

		//Mojo.Log.info("\n\nmonth-assistant.setup: timing: %j", this.timing);

		// Bind event handlers' "this" scope:
		this.showJumpTo			= this.showJumpTo			.bind (this);
		this.showCalendarPicker	= this.showCalendarPicker	.bind (this);
		this.onCalendarPicked	= this.onCalendarPicked		.bind (this);
		this.onDayTapped		= this.onDayTapped			.bind (this);
		this.onTodayTapped		= this.onTodayTapped		.bind (this);

		this.monthDate		= this.app.getCurrentDateTime();
		this.dvCurrentDay	= this.controller.get ("dv_current_day");
		this.month_days		= this.controller.get ("mv_days");
		this.reminders		= this.app.getReminderManager();

		this.reminders.observeReminders	("month", this);
		this.app.observeCurrentDateTime	("month", this);
		this.app.observeTimeChange		("month", this);
		this.app.observeDayChange		("month", this);
		this.eventManager.observeDatabaseChanges ("month", this);

		var prefs		= this.prefsManager.getPrefs();
		this.savedPrefs	= prefs ? Foundations.ObjectUtils.clone (prefs) : undefined;
		this._months	= this.month_days.appendChild (this._createEmptyMonthTable());

		this.setupScrolling (this._months);
		this.buildViewHeader();
		this.buildDayHeader();

		this.calendarsManager.observeCalendars ("month", this);
		this.prefsManager.observeCalendarPrefs ("month", this);

		this._createDummyWeek();

		this.menuModel = 
		{	visible	:	true
		,	items	:
			[	{}
			,	{	label		: $L('Views')
				,	toggleCmd	: 'month'
				,	items		:
					[	{	command	: 'day'
						,	icon	: 'menu-day'
						,	label	: $L('Day')
						}
					,	{	command	: 'week'
						,	icon	: 'menu-week'
						,	label	: $L('Week')
						}
					,	{	command	: 'month'
						,	icon	: 'menu-month'
						,	label	: $L('Month')
						}
					]
				}
			,	{}
			]
		};

		this.controller.setupWidget	(Mojo.Menu.commandMenu, undefined, this.menuModel);
		this.controller.setupWidget	(Mojo.Menu.appMenu, { omitDefaultItems:true }, this.appMenuModel);
		this.remindersUpdated();	// Update missed reminders menu item
	},

	setupScrolling: function setupScrolling (monthsElement) {
		var	direction	= 0
		,	scrolling	= false
		,	view		= this;

		function onFlick (event) {
			if (scrolling) {
				Mojo.Event.stop (event);
			}
		}

		function onMove (event) {
			scrolling = view.isScrolling = !event.scrollEnding;										//console.log("\n\n\nonMove: "+JSON.stringify({direction:direction, isScrolling:scrolling})+", event:\n\t"+JSON.stringify (event)+"\n\n\n");
			if (scrolling) { return; }
			view.app.setCurrentDateTime (view.app.getCurrentDateTime().addMonths (direction));
			direction = 0;
		}

		function onSnap (event) {
			direction = event.value - event.oldValue;	// Set direction based on snap index change.
			view.monthDate.addMonths (direction);		// Shift current month by direction.
			view.buildMonthTitle();																	//console.log("\n\n\nonSnap: ["+event.oldValue+"] -> ["+event.value+"], diff: ["+direction+"], month: ["+view.monthDate.getMonth()+"]\n\n\n");
		}

		var	container
		,	controller	= view.controller
		,	scrollData;

		view.handleScrolling = function handleScrolling() {
			controller		.listen (container, Mojo.Event.propertyChange, onSnap);
			container.mojo	.addMovementListener (onMove);
			scrollData		.observe (Mojo.Event.flick, onFlick);
			Mojo.Dom		.makePositioned (scrollData);
		};

		view.ignoreScrolling = function ignoreScrolling() {
			scrollData		.stopObserving (Mojo.Event.flick, onFlick);
			container.mojo	.removeMovementListener (onMove);
			controller		.stopListening (container, Mojo.Event.propertyChange, onSnap);
		};

		var model = { snapElements: { x:[], y:[] }, snapIndex: 1 };

		(function setupScrollContainer (monthsElement) {
			container	= controller.get ("mv_scroll_container");
			scrollData	= controller.get ("mv_scroll_data");

			if (monthsElement) {
				// Month days haven't been loaded as yet so default the snap elements to the 1st day
				// of every 6th week. Snap elements will later be updated to the 1st day of each
				// month in the 18-week data set.
				var weekRows = monthsElement.rows;

				model.snapElements.y =
				[	weekRows [((0	/*index of first day*/						+ 21) / 7) | 0]	// "| 0" drops remainder
				,	weekRows [((43	/*index of the first day in the 7th week*/	+ 21) / 7) | 0]
				,	weekRows [((85	/*index of the first day in the 13th week*/	+ 21) / 7) | 0]
				];
			}
			controller.setupWidget ("mv_scroll_container", { mode: "vertical-snap" }, model);
		})(monthsElement);

		view.updateScrollerModel = function updateScrollerModel (action) {
			//	Updates the scroller model as specified by:
			//	@param	action =
			//			{	snapElements: object	; optional
			//				{	y		: array		; vertical snap elements.
			//				,	notify	: boolean	; whether to send modelChanged notifications.
			//				}
			//			,	snapIndex	: object	; optional
			//				{	value	: number	; index of vertical snap element.
			//				,	notify	: boolean	; whether to send move and snap notifications.
			//				}
			//			}
			var modelChanged;

			if (!!action.snapElements && action.snapElements.y && action.snapElements.y.length) {
				for (var i=model.snapElements.y.length; i--;) {
					model.snapElements.y [i] = action.snapElements.y [i];
				}
				modelChanged = !!model.snapElements.notify;
			}

			if (!!action.snapIndex && !isNaN (action.snapIndex.value)) {
				(container.mojo.setSnapIndex
				(	action.snapIndex.value
				,	!!action.snapIndex.animate
				,	!action.snapIndex.notify
				));
			} else if (modelChanged) {
				controller.modelChanged (model);
			}
		};
	},//END: function setupScrolling (monthsElement)

	showDayDetail: function(day) {
		this.controller.showDialog({template: 'shared/daydetail', assistant: new DayDetailDialogAssistant(this,day)});
	},

	showJumpTo: function() {
		this.controller.showDialog ({ template: 'shared/jumpto', assistant: new JumptoDialogAssistant (this.controller) });
	},

	syncAllCalendars: function() {
		this.calendarsManager.syncAllCalendars (this.controller);
	},

	// TODO: Remove???  Accounts or the Sync framework should handle this ...
	syncAllCallback: function(response) {
		Mojo.Controller.appController.showBanner({
			messageText: $L("Syncing accounts")},
			null /*launchArguments*/,
			"calendar-sync-all");
	},
	
	handleCommand: function (event) 
	{
		if(event.type == Mojo.Event.command) {
			if (event.command == 'month') {
				Mojo.Event.stop (event);
				this.app.goToToday();
			} else if (event.command == 'week') {
				Mojo.Event.stop (event);
				this.handleWeekView();
			} else if (event.command == 'day') {
				Mojo.Event.stop (event);
				this.handleDayView();
			} else if (event.command == Mojo.Menu.prefsCmd) {
				Mojo.Event.stop (event);
				this.controller.stageController.pushScene('prefs');
			}else if(event.command == Mojo.Menu.helpCmd){
				this.app.getAppManagerService().launchHelp (this.controller);
			} else if (event.command == 'sync') {
				Mojo.Event.stop (event);
				this.syncAllCalendars();
			}else if (event.command == 'today') {
				Mojo.Event.stop (event);
				this.app.goToToday();
			}else if (event.command == 'jumpto') {
				Mojo.Event.stop (event);
				this.showJumpTo();
			} else if (event.command == 'reminders') {
				Mojo.Event.stop (event);
				this.controller.stageController.pushScene('reminder-list');
			}				
		}
		else if(event.type == Mojo.Event.commandEnable && event.command == Mojo.Menu.prefsCmd) {
			// Enable prefs menuitem for this scene.
			event.stopPropagation();
		}
	},

	cleanup: function() {
		//Mojo.Log.info('month-assistant: cleanup');

		this.reminders.stopObservingReminders('month');
		this.calendarsManager.stopObservingCalendars('month');
		this.prefsManager.stopObservingCalendarPrefs ("month");
		this.app.stopObservingCurrentDateTime("month");
		this.app.stopObservingTimeChange ("month");
		this.app.stopObservingDayChange  ("month");
		this.eventManager.stopObservingDatabaseChanges("month");
	},

	activate: function() {
///*DEBUG:*/	this.timing.activate = -new Date();

		//Mojo.Log.info ("\n\nmonth-assistant.activate\n\n");

		// Tap JumpTo:
		this.controller.get ('mv_view_header').observe (Mojo.Event.tap, this.showJumpTo);

		if(this.locale=="zh_cn"){
			this.daysHoldHandler = this.handleDaysHold.bindAsEventListener(this);
			this.month_days.observe('mojo-hold', this.daysHoldHandler);
		}

		// Tap Calendar Picker:
		this.controller.get ('mv_calendar_source').observe (Mojo.Event.tap, this.showCalendarPicker);

		// Tap Any Day:
		this.month_days.observe (Mojo.Event.tap, this.onDayTapped);

		// Tap Current Day:
		this.dvCurrentDay.observe (Mojo.Event.tap, this.onTodayTapped);

		this.handleScrolling();
		this.buildMonth();
///*DEBUG:*/	this.timing.launch += +new Date();

		// The colors have been updated, so we clear the cache and update everything
		if (this.pendingCalendarSettingsUpdate === true) {
			//this.loadBusyTimes ();
			// TODO: Clean this up when we've consolidated the cache stuff
			WeekAssistant.cacheSize = 0;
			WeekAssistant.weekCache = new Hash();
			WeekAssistant.cacheSubscribed = new Hash();
			DayAssistant.cacheSize = 0;
			DayAssistant.dayCache = new Hash();
			DayAssistant.cacheSubscribed = new Hash();
			this.pendingCalendarSettingsUpdate = false;
		}

///*DEBUG:*/	this.timing.activate += +new Date();
	},

	timeChangeUpdated: function() {
		//Mojo.Log.info('month-assistant:timeChangeUpdated');
		this.monthDate = this.app.getCurrentDateTime();
		this.buildMonth();
	},

	deactivate: function() {
		this.ignoreScrolling();

		this.month_days		.stopObserving (Mojo.Event.tap	, this.onDayTapped);
		this.dvCurrentDay	.stopObserving (Mojo.Event.tap	, this.onTodayTapped);

		this.controller.get ('mv_view_header')		.stopObserving (Mojo.Event.tap, this.showJumpTo);
		this.controller.get ('mv_calendar_source')	.stopObserving (Mojo.Event.tap, this.showCalendarPicker);
		
		if(this.locale=="zh_cn"){
			this.month_days.stopObserving('mojo-hold', this.daysHoldHandler);
		}
	},

	// *** EVENT HANDLERS ***

	onDayTapped: function (event) {
		if (this.isScrolling) { return; }

		//Mojo.Log.info ('month-assistant: onDayTapped');

		var targetDay = this.controller.get (event.target);

		if (targetDay.hasClassName ('monthview-day-numeral')) {
			targetDay = targetDay.parentNode;
		}

		//Mojo.Log.info ("onDayTapped: %s" , targetDay.id);

		if (targetDay.hasClassName ('day')) {
			var daycount= targetDay.parentNode.rowIndex * 7 + targetDay.cellIndex
			,	day		= new Date (this.firstDay);

			day.addDays (daycount);
			this.gotoDayView (day);
		}
	},
	
	handleDaysHold: function(event) {
		if(this._scrolling) return;
		var targetDay = this.controller.get(event.target);
		if (targetDay)
		{
			while(targetDay.tagName!="TD")
				targetDay = targetDay.parentNode;
			event.stop();
			var daycount = targetDay.parentNode.rowIndex * 7 + targetDay.cellIndex;
			var day = new Date(this.firstDay);
			day.addDays(daycount);
			this.showDayDetail(day);
		}
	},

	gotoDayView: function (day) {
		this.app.setCurrentDateTime (day);
		var sController = this.controller.stageController;
		sController.popScene();
		sController.pushScene({name: "day", transition: Mojo.Transition.crossFade, disableSceneScroller: true});
	},

	showCalendarPicker: function (event) {
		Mojo.Event.stop (event);

		var listItems = this.calendarsManager.buildCalendarsMenu (true /*includeAll*/, true /*includeReadOnly*/, true /*includeExcludedFromAll*/);

		this.controller.popupSubmenu (
		{	items			: listItems
		,	manualPlacement	: true
		,   onChoose		: this.onCalendarPicked
//		,	placeNear		: event.target
		,	popupClass		: "cal-selector-popup"
		,	toggleCmd		: this.calendarsManager.getCurrentCal()
		});
	},

	onCalendarPicked: function (value) {
		if (!value) { return; }

		if (value == "viewOptions"){
			this.controller.stageController.pushScene ("prefs-options");
			return;
		}

		this.calendarsManager.setCurrentCal (value);
		this.buildMonth();

		// TODO: Clean this up when we've consolidated the cache stuff
		WeekAssistant.cacheSize			= 0;
		WeekAssistant.weekCache			= new Hash();
		WeekAssistant.cacheSubscribed	= new Hash();
		DayAssistant.cacheSize			= 0;
		DayAssistant.dayCache			= new Hash();
		DayAssistant.cacheSubscribed	= new Hash();
	},

	// ** MENU **
	handleDayView: function (e) {
		//var today = new Date();
		//var currentDateTime = this.app.getCurrentDateTime();
		//if (currentDateTime.getMonth() == today.getMonth()) {
		//	this.app.goToToday();
		//} else {
		//	this.app.setCurrentDateTime (currentDateTime.moveToFirstDayOfMonth());
		//}
		//we should always show the day that was previously viewed
		//commenting out the above logic for bug /NOV-28956
		var sController = this.controller.stageController;
		sController.popScene();
		sController.pushScene({name: "day", transition: Mojo.Transition.crossFade, disableSceneScroller: true});
	},
  
	handleWeekView: function(e) {
		//var today = new Date();
		//var currentDateTime = this.app.getCurrentDateTime();
		//if (currentDateTime.getMonth() == today.getMonth()) {
		//	this.app.goToToday();
		///} else {
		//	this.app.setCurrentDateTime (currentDateTime.moveToFirstDayOfMonth());
		//}
		//commenting out the above logic for bug /NOV-28956
		var sController = this.controller.stageController;
		sController.popScene();
		sController.pushScene({name: "week", transition: Mojo.Transition.crossFade, disableSceneScroller: true});
	},

	showToday: function() {
		var style = this.dvCurrentDay.style;

		if (this.thisDayIndex < 0 || this.thisDayIndex > 125) {		// Today isn't in range so
			if (style.display != 'none') {							// if its indicator isn't hidden
				style.display = 'none';								// hide it.
			}
			return;
		}

		var	todayDiv	= this._months.rows [(this.thisDayIndex / 7) | 0].cells [this.thisDayIndex % 7]
		,	todayOffset	= todayDiv.positionedOffset();

		style.top		= todayOffset.top	+ "px";					// Set today indicator's vertical
		style.left		= todayOffset.left	+ "px";					// and horizontal positions
		style.display	= "block";									// then display it.
	},

	onTodayTapped: function (event) {
		this.gotoDayView (new Date());
	},

	remindersUpdated: function() {
		// TODO: Extract this and all other views' reminder handling to common component:
		// i.e. reminder-assistant*.js, new app.menu.js
		if (this.reminders.getNumReminders() === 0) {
			if (this.reminderMenuItemId == this.appMenuModel.items[4].id) {
				this.appMenuModel.items.splice (4, 1);
				this.controller.modelChanged (this.appMenuModel);
			}
		} else if (this.reminderMenuItemId != this.appMenuModel.items[4].id) {
			this.appMenuModel.items.splice (4, 0,
			{	command	: 'reminders'
			,	id		: this.reminderMenuItemId
			,	label	: $L('Missed reminders...')
			});
			this.controller.modelChanged (this.appMenuModel);
		}
	}
	
});
