this._root["__MojoFramework_contacts.ui"] = function(MojoLoader, exports, root) {


//@ sourceURL=contacts.ui/prologue.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global MojoLoader*/


var IMPORTS =  MojoLoader.require({
	name: "underscore",
	version: "1.0"
}, {
	name: "contacts",
	version: "1.0"
}, {
	name: "globalization",
	version: "1.0"
}, {
	name: "foundations",
	version: "1.0"
}, {
	name: "accounts.ui",
	version: "1.0"
});

var _ = IMPORTS.underscore._,
	Contacts = IMPORTS.contacts,
	Foundations = IMPORTS.foundations,
	AccountsLib = IMPORTS["accounts.ui"],
	Assert = Foundations.Assert,
	Future = Foundations.Control.Future,
	PalmCall = Foundations.Comms.PalmCall,
	LIB_ROOT = MojoLoader.root,
	Globalization = IMPORTS.globalization.Globalization;

var resourceBundleFactory = new Globalization.ResourceBundleFactory(MojoLoader.root);
var RB = resourceBundleFactory.getResourceBundle();

AccountsLib.enableExtensions();

//@ sourceURL=contacts.ui/PersonListWidget.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Assert, Mojo, _, console, Contacts, Utils, EventListenerManager, 
pushPeoplePickerScene, RB, LIB_ROOT, PersonAsyncDecorator, Foundations, Globalization */

var PersonListWidget = exports.PersonListWidget = function () {
	Assert.requireDefined(Mojo, "PersonListWidget: Failed to detect the Mojo framework");
	
	var _sceneController,
		_div,
		_model,
		_uniqueId = "PFLW_" + Date.now(), // DOM elements for this widget that have ids are prefixed with a unique id that is specific to this scene instance
		_stylesheets = [Utils.getStylesheetPath("common.css"), Utils.getStylesheetPath("list.css")],
		_eventListenerManager = new EventListenerManager(),
		_personListScrollPosition,
		_favoriteListScrollPosition,
		_personListDataSource,
		_favoriteListDataSource,
		_personListElement,
		_favoriteListElement,
		_currentView,
		_filterString = "",
		
		
		/**
		* Returns the full id for the specified human readable id
		* @param {String} id
		* @return {String}
		*/
		getWidgetElementId = function (id) {
			return (_uniqueId + id);
		},
		
		/**
		 * Returns the DOM element for the specified human readable id 
		 * @param {String} id
		 * @return {String}
		 */
		getWidgetElement = function (id) {
			return _sceneController.get(getWidgetElementId(id));
		},
		
		transformListItem = function (person) {
			if (!person) {
				return person;
			}
			
			//var start = Date.now();
			person = Contacts.PersonFactory.createPersonDisplayLite(person, _model.sortOrder);
			//console.log("Took " + (Date.now()-start) + "ms to create/transform person object");
			
			// Hide list elements that should be excluded.  Example: when you are manually linking
			// from the detail view, you do not want to show an entry in the list for the person object
			// that you originated from
			if (_model.exclusions && Array.isArray(_model.exclusions)) {
				if (_model.exclusions.indexOf(person._id) !== -1) {
					person.exclude = "exclude";
				}
			}
			
			return person;
		},
		
		dividerLabelCallback = function (person) {
			if (person && !person.exclude) {
				//contacts library - javascript/PersonDisplayLite
				return person.dividerText;
			} else {
				return "";
			}
		},
		
		personDecorator = function (itemModel) {
			var decorations = {},
				displayName = Foundations.StringUtils.escapeHTML(itemModel.displayName);
			
			if (_filterString) {
				// TODO: perf - should cache filter match patterns between invocations
				decorations.displayName = Mojo.PatternMatching.addContactMatchFormatting(displayName, _filterString);
			} else {
				decorations.displayName = displayName;
			}
			
			return decorations;
		},
		
		createAsyncPersonDecorator = function (items, callback) {
			return new PersonAsyncDecorator(items, callback);
		},
				
		setupPersonList = function () {
			var dataSourceAssistant = Contacts.ListWidget.getTypedownDBDataSourceAssistant({
					favoritesOnly: _model.favoritesOnly,
					excludeFavorites: _model.excludeFavorites
				}),
				wrappedDataSourceAssistant = Utils.formatDataSourceAssistant(dataSourceAssistant, transformListItem),
				dividerTemplate;
			
			_personListDataSource = new Mojo.DataSource(wrappedDataSourceAssistant);
			_personListElement = getWidgetElement("person-list");
			_model.personListTapCallback = _model.personListTapCallback || _model.listTapCallback;
			_model.personListDeleteCallback = _model.personListDeleteCallback;
			
			if (_model.sortOrder === Contacts.ListWidget.SortOrder.firstLast || _model.sortOrder === Contacts.ListWidget.SortOrder.lastFirst) {
				dividerTemplate = Utils.getTemplatePath("person-filter-list-widget/group-separator");
			} else {
				dividerTemplate = Utils.getTemplatePath("person-filter-list-widget/multiline-separator");
			}
			
			if(_model.personListDeleteCallback && _.isFunction(_model.personListDeleteCallback))
				var allowDelete=true;
			else
				var allowDelete=false;

			_sceneController.setupWidget(getWidgetElementId("person-list"), {
				templateRoot: LIB_ROOT,
				templates: {
					item: Utils.getTemplatePath("person-filter-list-widget/person-item"),
					empty: Utils.getTemplatePath("person-filter-list-widget/empty")
				},
				uniquenessProperty: "_id",
				dataSource: _personListDataSource,
				dividers: {
					labelCallback: dividerLabelCallback,
					template: dividerTemplate
				},
				decorator: personDecorator,
				asyncDecorator: createAsyncPersonDecorator,
				swipeDelete: allowDelete
			}, {});
			
			if (_model.personListTapCallback && _.isFunction(_model.personListTapCallback)) {
				_eventListenerManager.addListener(_personListElement, Mojo.Event.listTap, _model.personListTapCallback);
			} else {
				console.warn("PersonListWidget (setupPersonList) was set up without a listTapCallback.  Do you really want to do this?");
			}

			if(_model.personListDeleteCallback && _.isFunction(_model.personListDeleteCallback))
				_eventListenerManager.addListener(_personListElement,Mojo.Event.listDelete,_model.personListDeleteCallback);
		},
		
		handleAddFavorite = function (person) {
			person.makeFavorite(true); // pass true so it saves immediately
			// TODO: could add a UI hack to inject this favorite into the list for fast feedback while the DB updates & fires the watch
		},
		
/*		handleAddFavoriteTap = function (event) {
			pushPeoplePickerScene(_sceneController.stageController, {
				excludeFavorites: true,
				iconClass: "icon-add-fav",
				message: RB.$L("Add to Favorites"),
				callback: handleAddFavorite
			});
		},
*/
		handleFavoriteDelete = function (items, callback) {
			items.forEach(function (item) {
				var person = Contacts.PersonFactory.createPersonDisplay(item);
				person.unfavorite(true);
			});
			
			callback();
		},
		
		setupFavoriteList = function () {
			var dataSourceAssistant = Contacts.ListWidget.getTypedownDBDataSourceAssistant({
					favoritesOnly: true
				}),
				wrappedDataSourceAssistant = Utils.formatDataSourceAssistant(dataSourceAssistant, transformListItem);
			
			_favoriteListDataSource = new Mojo.DataSource(wrappedDataSourceAssistant);
			_favoriteListElement = getWidgetElement("favorite-list");
			_model.favoriteListTapCallback = _model.favoriteListTapCallback || _model.listTapCallback;
			
			//monkey-patch the removeItems so that it doesn't actually delete the person - instead, we just unfavorite it (NOV-115844)
			_favoriteListDataSource.removeItems = handleFavoriteDelete;
			
			_sceneController.setupWidget(getWidgetElementId("favorite-list"), {
				templateRoot: LIB_ROOT,
				templates: {
					item: Utils.getTemplatePath("person-filter-list-widget/person-item")
				},
				uniquenessProperty: "_id",
				dataSource: _favoriteListDataSource,
				decorator: personDecorator,
				asyncDecorator: createAsyncPersonDecorator,
/*				addItem: {
					label: RB.$L("Add Favorite...")
				},
*/
				swipeDelete: {
					deleteText: RB.$L("Remove")
				}
			}, {});
			
			if (_model.favoriteListTapCallback && _.isFunction(_model.favoriteListTapCallback)) {
				_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listTap, _model.favoriteListTapCallback);
			} else {
				console.warn("PersonListWidget (setupFavoriteList) was set up without a listTapCallback.  Do you really want to do this?");
			}
			
//			_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listAdd, handleAddFavoriteTap);
			_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listDelete, handleFavoriteDelete);
		},
		
		swapLists = function (elementIdToShow, elementIdToHide, useTransition, state) {
			var elementToShow = getWidgetElement(elementIdToShow),
				elementToHide = getWidgetElement(elementIdToHide),
				transition;
			
			if (useTransition) {
				transition = _sceneController.prepareTransition(Mojo.Transition.crossFade, false);
			}
			
			Mojo.Dom.show(elementToShow);
			Mojo.Dom.hide(elementToHide);
			_sceneController.showWidgetContainer(elementToShow);
			_sceneController.hideWidgetContainer(elementToHide);
			

			if (state) {
				_sceneController.sceneScroller.mojo.setState(state);
			}
			
			
			if (transition) {
				transition.run();
			}
		};
	
	return {
		/**
		* 
		* @param {Object} div
		* @param {Object} sceneController
		* @param {Object} model 
		*					{
		*						mode: {string},
		*						excludeFavorites: {boolean}
		*						listTapCallback: {Function},			// sets the callback for both lists
		*						personListTapCallback: {Function},
		*						favoriteListTapCallback: {Function},
		*						exclusions: {Array} // array of ids,
		*						sortOrder: One of Contacts.ListWidget.SortOrder.XXX
		*					}
		*/
		setup: function (sceneController, div, model) {
			Assert.requireFalse(_sceneController, "PersonListWidget setup() has already been called! Aborting.");
			Assert.require(sceneController, "PersonListWidget requires a sceneController");
			_sceneController = sceneController;
			_div = (div && _.isString(div) ? sceneController.get(div) : div);
			_model = model || {};
			_model.mode = _model.mode || PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST;
			Assert.require(div, "PersonListWidget requires a div element");
			
			if (!_model.sortOrder) {
				Mojo.Log.warning("PersonListWidget: no sort order provided!  Defaulting to Contacts.ListWidget.SortOrder.defaultSortOrder");
				_model.sortOrder = Contacts.ListWidget.SortOrder.defaultSortOrder;
			}
			
			Utils.loadSceneStyles(_sceneController, _stylesheets, _uniqueId);
			
			// inject widget specific HTML into the div supplied by the caller
			var listWidgetModel = {
					uniqueId: _uniqueId
					// if we are only displaying favorites, hide the favorite icon on each list item by
					// setting this class on the outer list widget element.  The CSS on the list item will look for this
					//favoritesListClass: (_model.favoritesOnly ? Contacts.PersonDisplay.FAVORITES_LIST_CLASS : "")
				},
				hasPersonList = false,
				hasFavoriteList = false;
				
			div.innerHTML = Mojo.View.render({object: listWidgetModel, templateRoot: LIB_ROOT, template: Utils.getTemplatePath("person-filter-list-widget/widget-template")});
			
			if (_model.mode === PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST || _model.mode === PersonListWidget.MODE.PERSON_LIST) {
				setupPersonList();
				hasPersonList = true;
			}
			if (_model.mode === PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST || _model.mode === PersonListWidget.MODE.FAVORITE) {
				setupFavoriteList();
				hasFavoriteList = true;
			}
			
			if (_model.defaultView === PersonListWidget.VIEW.PERSON_LIST) {
				this.showPersonList();
			} else if (_model.defaultView === PersonListWidget.VIEW.FAVORITE_LIST) {
				this.showFavoriteList();
			} else {
				// pick a default
				if (hasPersonList) {
					this.showPersonList();
				} else if (hasFavoriteList) {
					this.showFavoriteList();
				}
			}
		},

		getCurrentView: function () {
			return _currentView;
		},
		
		teardown: function () {
			Utils.unloadSceneStyles(_sceneController, _stylesheets, _uniqueId);
			_eventListenerManager.destroyListeners();
		},
		
		showPersonList: function (useTransition) {
			if (_currentView === PersonListWidget.VIEW.PERSON_LIST) {
				return;
			}
			
			_currentView = PersonListWidget.VIEW.PERSON_LIST;
			_favoriteListScrollPosition = _sceneController.sceneScroller.mojo.getState();
			swapLists("person-list-container", "favorite-list-container", useTransition, _personListScrollPosition);
		},
		
		showFavoriteList: function (useTransition) {
			if (_currentView === PersonListWidget.VIEW.FAVORITE_LIST) {
				return;
			}
			_currentView = PersonListWidget.VIEW.FAVORITE_LIST;
			_personListScrollPosition = _sceneController.sceneScroller.mojo.getState();
			
			swapLists("favorite-list-container", "person-list-container", useTransition, _favoriteListScrollPosition);
		},
		
		showAddFavorite: function () {
			pushPeoplePickerScene(_sceneController.stageController, {
				excludeFavorites: true,
				iconClass: "icon-add-fav",
				message: RB.$L("Add to Favorites"),
				callback: handleAddFavorite
			});
		},

		// Filters the visible list
		filter: function (str, setCountFn) {
			var dataSource,
				listElement;
			if (_currentView === PersonListWidget.VIEW.FAVORITE_LIST) {
				dataSource = _favoriteListDataSource;
				listElement = _favoriteListElement;
			} else {
				dataSource = _personListDataSource;
				listElement = _personListElement;
			}
			
			_filterString = str;
			dataSource.setFilterString(str);
			dataSource.getCount(setCountFn);
			listElement.mojo.invalidate();
		},
		
		invalidate: function () {
			//TODO: this can be called when the sort order changes, but we don't update the divider template based on the new sort order!
			
			_favoriteListElement.mojo.invalidate();
			_personListElement.mojo.invalidate();
		}
	};
};

PersonListWidget.MODE = {
	PERSON_LIST: "mode_person_list",
	FAVORITE_LIST: "mode_favorite_list",
	PERSON_AND_FAVORITE_LIST: "mode_person_and_favorite_list"
};

PersonListWidget.VIEW = {
	PERSON_LIST: "view_person_list",
	FAVORITE_LIST: "view_favorite_list"
};


//@ sourceURL=contacts.ui/DetailWidget.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Assert, Mojo, _, console, Contacts, Utils, EventListenerManager, MultipleListManager, TinyList,
RB, pushPeoplePickerScene, Future, pushReminderScene, OpenWithMenu, LIB_ROOT, PalmCall, AccountList, ImAddressStatusWatcher */

var DetailWidget = exports.DetailWidget = function () {
	Assert.requireDefined(Mojo, "DetailWidget: Failed to detect the Mojo framework");
	this.controller = undefined;
	this._widget = {
		div: undefined,
		model: undefined,
		stylesheets: [Utils.getStylesheetPath("common.css"), Utils.getStylesheetPath("detail.css"), Utils.getStylesheetPath("sandbox-styles.css")],
		uniqueId: "PFLW_" + Date.now()
	};
	
	this.person = undefined;
	this.personId = undefined;	// Need to have the id available as soon as possible. It may take a few moments for the decorated person to be created due to async DB calls
	
	this.personRev = undefined;
	this.clippedModel = {};
	this.currentlyLinking = false;
	this.listConfigs = [{
		listName: "phone",
		listId: "phoneList",
		itemTemplate: "phone-item"
	}, {
		listName: "email",
		listId: "emailList",
		itemTemplate: "email-item"
	}, {
		listName: "messaging",
		listId: "messagingList",
		itemTemplate: "messaging-item"
	}, {
		listName: "address",
		listId: "addressList",
		itemTemplate: "address-item"
	}, {
		listName: "url",
		listId: "urlList",
		itemTemplate: "url-item"
	}, {
		listName: "notes",
		listId: "notesList",
		itemTemplate: "notes-item"
	}, {
		listName: "moredetails",
		listId: "moredetailsList",
		itemTemplate: "moredetails-item"
	}];
	
	this.eventListenerManager = new EventListenerManager();
	this.listManager = new MultipleListManager();
	
	this.imStatusWatchers = {};
};
	
/**
* 
* @param {Object} div
* @param {Object} sceneController
* @param {Object} model
*					{
*						person: {PersonDisplay},				// decorated PersonDisplay object
*						contactsFuture: {Future},				// the result of this future contains an array of decorated contacts
*																// contactsFuture is only used when passing a decorated person object as well
*						personId: {String},						// The widget will fetch & create the decorated PersonDisplay + linked contacts
*						onPersonRenderedCallback: {Function},		// executed when the person object has been rendered (this is before the contacts have been rendered in the drawer)
*						onRenderAfterUpdateCallback: {Function},	// executed when the mojodb watch fires
*						hideLinkedContacts: {boolean}
*					}
*/
DetailWidget.prototype.setup = function (sceneController, div, model, accountList) {
	var widgetSetupFuture = new Future();
	
	Assert.requireFalse(this.controller, "DetailWidget setup() has already been called! Aborting.");
	Assert.require(sceneController, "DetailWidget requires a sceneController");
	this.controller = sceneController;
	this._widget.div = (div && _.isString(div) ? this.controller.get(div) : div);
	this._widget.model = model || {};
	this.accountList = accountList || new AccountList();
	this.boundAccountsUpdatedFunction = this.accountsUpdated.bind(this);
	this.accountList.addListener(this.boundAccountsUpdatedFunction);
	
	Assert.require(div, "DetailWidget requires a div element");
	Assert.require((this._widget.model.person || this._widget.model.personId), "DetailWidget requires a personId or a decorated personDisplay");
	
	Utils.loadSceneStyles(this.controller, this._widget.stylesheets, this._widget.uniqueId);
	
	// inject widget specific HTML into the div supplied by the caller
	this._widget.div.innerHTML = Mojo.View.render({object: {uniqueId: this._widget.uniqueId}, templateRoot: LIB_ROOT, template: Utils.getTemplatePath("detail-widget/widget-template")});
	
	/*
	 * Handle the params set in model
	 */
	//first, we need to know whether or not to show the skype calling icons
	if (this._widget.model.skypeCallingEnabled !== undefined) {
		this.skypeCallingEnabled = this._widget.model.skypeCallingEnabled;
		//do this just to kick off the chain
		widgetSetupFuture.now(this, function () {
			return true;
		});
	} else {
		//check for a skype voice account template so we know whether to show the skype call icons in detail view
		widgetSetupFuture.now(this, function () {
			return PalmCall.call("palm://com.palm.service.accounts/", "listAccountTemplates", {
				capability: "VOICE"
			});
		});
		widgetSetupFuture.then(this, function () {
			var result = widgetSetupFuture.result;
			
			//if something goes wrong, default to false
			if (!result || !result.results || !_.isArray(result.results)) {
				this.skypeCallingEnabled = false;
				return true;
			}
			
			this.skypeCallingEnabled = result.results.some(function (accountTemplate) {
				return accountTemplate.templateId === "com.skype";
			});
			
			return true;
		});
	}
	
	if (this._widget.model.person) {
		widgetSetupFuture.then(this, function () {
			var dummy = widgetSetupFuture.result;
			
			return this._widget.model.person;
		});
	} else if (this._widget.model.personId) {
		this.personId = this._widget.model.personId;
		widgetSetupFuture.then(this, function () {
			var dummy = widgetSetupFuture.result;
			
			return Contacts.Person.getDisplayablePersonAndContactsById(this._widget.model.personId);
		});
	}
	
	// Fetch DOM elements
	this.personDetailElement = this.controller.get('contact_detail');
	this.header = this.controller.get('detail-header');
	this.headerDrawer = this.controller.get("clipped-info-drawer");
	this.linkedContactsDiv = this.controller.get('clipped-list');
	this.linkMoreButton = this.controller.get('clip-button');
	this.reminderContainer = this.controller.get("reminderContainer");
	
	this.controller.setupWidget("clipped-info-drawer", {
		open: false
	}, {});
	
	//////////////////////////////////
	// setup the linked contacts list
	//////////////////////////////////
	this.clippedModel = {
		items: []
	};
	this.clippedList = {
		templateRoot: LIB_ROOT,
		itemTemplate: Utils.getTemplatePath("detail-widget/clipped-item")
	};
	this.controller.setupWidget('clipped-list', this.clippedList, this.clippedModel);
	
	if (!this._widget.model.hideLinkedContacts) {
		this.linkedContactsTapDestructor = TinyList.setupTapHandler(this.linkedContactsDiv, this.clippedModel, this.handleLinkedContactsListTap.bind(this));
		
		this.eventListenerManager.addListener(this.header, Mojo.Event.tap, this.toggleLinked.bind(this));
		this.eventListenerManager.addListener(this.linkMoreButton, Mojo.Event.tap, this.handleLinkMore.bind(this));
	}	
	//////////////////////////////////
	//////////////////////////////////
	
	// create models + setup widgets for lists (PhoneNumbers, EmailAddresses, etc...)
	this.setupLists();
	
	// render the header and all lists
	widgetSetupFuture.then(this, function () {
		var person = widgetSetupFuture.result,
			contacts;
		
		// set + render the person
		this.setCurrentPerson(person);
		this.renderPerson();
		
		// get the contacts associated with the person
		if (this._widget.model.contactsFuture) {
			return this._widget.model.contactsFuture;
		} else {
			contacts = person.getContacts();
			if (_.isArray(contacts) && contacts.length > 0) {
				return contacts;
			} else {
				// fetch the contacts for this person
				return person.reloadContacts(Contacts.ContactFactory.ContactType.DISPLAYABLE);
			}
			
		}
	});
	
	// set + render the linked contacts
	widgetSetupFuture.then(this, function () {
		var contacts = widgetSetupFuture.result;
		this.setAndRenderLinkedContactsList(contacts);
		
		return true;
	});
	
	// Open with object for handling opening specific applications for the physical addresses
	this.openWith = new OpenWithMenu({
		isMimeType: false,
		urlOrMime: "mapto:",
		controller: this.controller,
		listId: "addressList"
	});
	
	this.eventListenerManager.addListener(this.reminderContainer, Mojo.Event.tap, this.handleReminderTap.bind(this));
	
	//TODO: all the other parts of this method that aren't in future.thens probably should be
	return widgetSetupFuture;
};

DetailWidget.prototype.accountsUpdated = function () {
	this.renderLinkedContactsList();
};

DetailWidget.prototype.teardown = function () {
	Utils.unloadSceneStyles(this.controller, this._widget.stylesheets, this._widget.uniqueId);
	this.eventListenerManager.destroyListeners();
	if (this.linkedContactsTapDestructor) {
		this.linkedContactsTapDestructor();
	}
	this.listManager.destroy();
	this.openWith.destroy();
	
	//now clean up the im status watchers
	var that = this;
	Object.keys(this.imStatusWatchers).forEach(function (imStatusWatcherKey) {
		that.imStatusWatchers[imStatusWatcherKey].cancel();
	});
	
	this.accountList.removeListener(this.boundAccountsUpdatedFunction);
	
	if (this.personWatchFuture) {
		this.person.stopWatchingForDatabaseChanges(this.personWatchFuture);
	}
};
	
DetailWidget.prototype.getPerson = function () {
	return this.person;
};

DetailWidget.prototype.getPersonId = function () {
	return this.personId;
};

DetailWidget.prototype.isCurrentlyLinking = function () {
	return this.currentlyLinking;
};
	
DetailWidget.prototype.fireRenderAfterUpdateCallback = function () {
	if (_.isFunction(this._widget.model.onRenderAfterUpdateCallback)) {
		this._widget.model.onRenderAfterUpdateCallback();
	}
};

DetailWidget.prototype.firePersonRenderedCallback = function () {
	if (_.isFunction(this._widget.model.onPersonRenderedCallback)) {
		this._widget.model.onPersonRenderedCallback();
	}
};
	
/**
* Returns the full id for the specified human readable id
* @param {String} id
* @return {String}
*/
DetailWidget.prototype.getWidgetElementId = function (id) {
	return (this._widget.uniqueId + id);
};

/**
 * Returns the DOM element for the specified human readable id 
 * @param {String} id
 * @return {String}
 */
DetailWidget.prototype.getWidgetElement = function (id) {
	return this.controller.get(this.getWidgetElementId(id));
};

DetailWidget.prototype.setupLists = function () {
	var that = this;
	this.listConfigs.forEach(function (listConfig) {
		var attributes = {
				itemTemplate: Utils.getTemplatePath("detail-widget/" + listConfig.itemTemplate),
				listTemplate: Utils.getTemplatePath("detail-widget/group-container-unlabeled"),
				templateRoot: LIB_ROOT				
			},
			model = {
				items: []
			},
			div = that.controller.get(listConfig.listId),
			updateFn = TinyList.renderData.bind(that, div, attributes),
			tapEventListenerDestructor = TinyList.setupTapHandler(div, model, that.handleListTap.bind(that));
		
		that.listManager.addListConfig(listConfig.listName, attributes, model, div, updateFn, tapEventListenerDestructor);
	});
};

// The person will be temporary when we construct a person object
// from the raw person data that is passed in from the list view
// We do not want to setup a database watch for this temporary object
DetailWidget.prototype.setCurrentPerson = function (person, isTemporary) {
	this.person = person;
	this.personId = this.person.getId();
	this.personRev = this.person.getRev();
	// observe database changes so we can update the person
	if (!isTemporary && person.getId()) {
		this.createPersonWatch();
	}
};

DetailWidget.prototype.createPersonWatch = function () {
	if (this.personWatchFuture) {
		this.person.stopWatchingForDatabaseChanges(this.personWatchFuture);
	}
	this.personWatchFuture = this.person.watchForDatabaseChanges().then(this, this.handlePersonDBUpdate);
};

DetailWidget.prototype.handlePersonDBUpdate = function (future) {
	var result = future.result,
		person,
		getContactsFuture;
	
	if (result.fired) {
		console.log("PERSON WATCH FIRED!!");
		this.createPersonWatch();
	} else {
		future.then(this, this.handlePersonDBUpdate);
		console.log("WATCH FUTURE RESULT: " + JSON.stringify(result));
		
		// create a decorated person object from the raw person object retrieved
		// from setting the watch
		person = Contacts.PersonFactory.createPersonDisplay(result);
		
		// make sure this person has a larger rev than the old person
		if (person.getRev() > this.personRev) {
			// fetch the contacts so we can re-render the linked contacts drawer 
			getContactsFuture = person.reloadContacts(Contacts.ContactFactory.ContactType.DISPLAYABLE);
			getContactsFuture.then(this, function () {
				var result = getContactsFuture.result;
				
				console.log("finished fetching contacts: " + person.getContacts().toString());
				this.setCurrentPerson(person);
				this.renderPersonAndLinkedContacts();
				this.fireRenderAfterUpdateCallback();
				
				getContactsFuture.result = true;
			});	
		} else {
			console.log("New person has a revId that is not greater than the currently rendered person");
		}
	}
};

DetailWidget.prototype.renderPerson = function () {
	this.renderHeader();
	this.renderContactPropertyLists();
	this.showDeferrals();
	this.renderReminder();
	
	this.firePersonRenderedCallback();
};

DetailWidget.prototype.renderLinkedContacts = function () {
	this.setAndRenderLinkedContactsList(this.person.getContacts());
};

DetailWidget.prototype.renderPersonAndLinkedContacts = function () {
	this.renderPerson();
	this.renderLinkedContacts();
};

DetailWidget.prototype.renderHeader = function (dontReloadPhotos) {
	if (this.person) {
		this.renderHeaderHelper();
		
		if (!dontReloadPhotos) {
			//TODO: can this be done more efficiently?  do we really have to render this twice?
			var future = this.person.getPhotos().getPhotoPath(Contacts.PersonPhotos.TYPE.SQUARE);
			future.then(this, function () {
				var newHeaderPhotoPath = future.result;
				
				if (newHeaderPhotoPath !== this.person.headerPhotoPath) {
					//now that we have a correct photo path, if it's different, render the header again
					this.renderHeaderHelper();
				}
			});
		}
	} else {
		console.log("Aborting renderHeader because this.person is null");
	}
};

DetailWidget.prototype.renderHeaderHelper = function () {
	this.header.innerHTML = Mojo.View.render({
		templateRoot: LIB_ROOT,
		template: Utils.getTemplatePath("detail-widget/header"),
		object: this.person
	});
};

DetailWidget.prototype.setAndRenderLinkedContactsList = function (contacts) {
	if (!this._widget.model.hideLinkedContacts) {
		this.clippedModel.items = contacts;
		this.renderLinkedContactsList();
	}
};

// This is used when temporarily injecting a fake contact just after a manual link
DetailWidget.prototype.appendLinkedContact = function (contact) {
	this.clippedModel.items.push(contact);
	this.renderLinkedContactsList();
};

DetailWidget.prototype.renderLinkedContactsList = function () {
	var that = this,
		future;
	
	if (this.clippedModel.items.length === 1) {
		this.clippedModel.items[0].singleItem = "x";
	} else if (this.clippedModel.items.length) {
		this.clippedModel.items[0].singleItem = "";
	}
	
	if (this.clippedModel.items.length > 0) {
		this.clippedModel.items[0].primary = "primary";
	}
	
	this.clippedModel.items.forEach(function (item) {
		item.iconPath = that.accountList.getAccountIcon(item.getAccountId().getValue(), item.getKindName(true));
	});
	
	TinyList.renderData(this.linkedContactsDiv, this.clippedList, this.clippedModel.items);
	
	//TODO: can this be done more efficiently?  do we really have to render this twice?
	future = Contacts.ContactPhoto.attachPhotoPathsToContacts(this.clippedModel.items, Contacts.ContactPhoto.TYPE.SQUARE, "drawerPhotoPath");
	future.then(this, function () {
		var dummy = future.result;
		
		//now that we have the photo paths, render the drawer again
		TinyList.renderData(this.linkedContactsDiv, this.clippedList, this.clippedModel.items);
	});
};

DetailWidget.prototype.handleLinkedContactsListTap = function (event) {
	var choices = [],
		contact = event.item,
		itemNode = event.itemElement;
		
	if (contact.isFakeContact || this.person.getContacts().length < 2) {
		return;
	}

	choices.push({
		label: RB.$L('Unlink this profile'),
		value: 'UNCLIP'
	});
	
	// Hide the set as primary if it is already the primary
	if (this.person.getContactIds().getArray()[0].getValue() !== contact.getId()) {
		choices.push({
			label: RB.$L('Set as primary profile'),
			value: 'MAKE_PRIMARY'
		});
	}
	
	// Hide delete profile option if the contact is read only
	if (!this.accountList.isContactReadOnly(contact)) {
		choices.push({
			label: RB.$L('Delete this profile'),
			value: 'DELETE_PROFILE',
			type: 'negative'
		});
	}
	choices.push({
		label: RB.$L('Cancel'),
		value: 'CANCEL',
		type: 'dismiss'
	});
	
	this.controller.showAlertDialog({
		onChoose: this.onClippedChoose.bind(this, contact, itemNode),
		title: contact.displayName,
		choices: choices
	});
};

DetailWidget.prototype.linkExisting = function (personDisplay) {
	var fakeContact = Contacts.ContactFactory.createContactDisplay();
	fakeContact.populateFromPerson(personDisplay);
	fakeContact.generateDisplayParams();
	
	this.appendLinkedContact(fakeContact);
	
	this.currentlyLinking = true;
	
	Contacts.Person.manualLink(this.person.getId(), personDisplay.getId()).then(this, function (future) {
		var result = future.result;
		// TODO: originally we forced a reload of the person/contact data.  Let's try just waiting for the data to bubble up from the DB
		//this.onSynchronousChange();
	});
};

DetailWidget.prototype.handleLinkMore = function () {
	var headerMessage,
		headerTemplate;
	
	if (this.isPseudoCard) {
		headerMessage = RB.$L("Link To Existing Contact");
	} else {
		headerMessage = RB.$L("Link to #{name}");
	}
	headerTemplate = new Mojo.Format.Template(headerMessage);
	headerMessage = headerTemplate.evaluate({
		name: this.person.displayName
	});
	
	pushPeoplePickerScene(this.controller.stageController, {
		exclusions: [this.person.getId()],
		iconClass: "icon-clip",
		message: headerMessage,
		callback: this.linkExisting.bind(this)
	});
};

DetailWidget.prototype.setFavoriteAppearance = function (newFavoriteState) {
	this.person.toggleFavoriteAppearance(newFavoriteState);
	this.renderHeaderHelper();
};

DetailWidget.prototype.onSynchronousChange = function () {
	// a synchronous change is one triggered by the UI
	// the detail scene is subscribed on linked contacts, but only in the 'new' way so updates may not fire immediately.
	// Refresh the detail scene manually when the source of the change was the UI (it's more important than refreshing immediately for backend changes).
	
	// this scene should also notify the list scene so it picks up the change immediately
	// (as opposed to treating it as a backend-initiated change, which could cause it to delay the UI refresh)
	//this.setupContact();
//	if (AppAssistant.listResetCallback) {
//		AppAssistant.listResetCallback();
//	}
};

/*
 * This should render all of the contact lists referenced in this.listConfigs.
 */
DetailWidget.prototype.renderContactPropertyLists = function () {
	var that = this,
		phoneNumberItems,
		messagingItems;
	
	/*
	 * emails, addresses, urls, notes, and more details are simple
	 */
	this.listManager.updateListItems("email", this.person.getEmails().getArray());
	this.listManager.updateListItems("address", this.person.getAddresses().getArray());
	this.listManager.updateListItems("url", this.person.getUrls().getArray());
	this.listManager.updateListItems("notes", this.person.getNotes().getArray());
	this.listManager.updateListItems("moredetails", this.generateMoreDetailsItems());
	
	/*
	 * before passing in the phone items, tack on the hasSpeedDial property for those phone numbers that have speed dials
	 */
	phoneNumberItems = this.person.getPhoneNumbers().getArray();
	phoneNumberItems.forEach(function (phoneNumberItem) {
		phoneNumberItem.hasSpeedDial = (phoneNumberItem.getSpeedDial()) ? "show-quick-dial" : "";
	});
	this.listManager.updateListItems("phone", phoneNumberItems);
	
	/*
	 * before passing in the messaging items, if skype calling is enabled, tack on the skypeIcon 
	 * property for those IM addresses that are of type skype
	 * this is the icon that we use to cross-launch the phone app instead of messaging
	 */
	messagingItems = this.person.getIms().getArray();
	messagingItems.forEach(function (messagingItem) {
		var hashKey,
			imStatusWatcher;
		
		//create IMStatus watchers here for those items that we don't already have one for
		hashKey = messagingItem.getNormalizedHashKey();
		if (!that.imStatusWatchers[hashKey]) {
			imStatusWatcher = new ImAddressStatusWatcher(messagingItem.getValue(), messagingItem.getType(), that.imStatusChangedCallback.bind(that));
			that.imStatusWatchers[hashKey] = imStatusWatcher;
		}
		
		messagingItem.imStatusClassName = that.imStatusWatchers[hashKey].getCurrentStatus().imStatusClassName;
		
		if (that.skypeCallingEnabled && messagingItem.getType() === Contacts.IMAddress.TYPE.SKYPE) {
			messagingItem.skypeIconDisabled = "";
		} else {
			messagingItem.skypeIconDisabled = "hide-skype";
		}
	});
	this.listManager.updateListItems("messaging", messagingItems);
};

DetailWidget.prototype.imStatusChangedCallback = function (newImStatus) {
	var messagingItems = this.person.getIms().getArray(),
		messagingItem = _.detect(messagingItems, function (imAddress) {
			//TODO: add case insensitivity
			return newImStatus.imAddress === imAddress.getValue() && newImStatus.type === imAddress.getType();
		});
	
	if (messagingItem) {
		messagingItem.imStatusClassName = newImStatus.imStatusClassName;
		this.listManager.updateListItems("messaging", messagingItems);
	}
};

DetailWidget.prototype.generateMoreDetailsItems = function () {
	var moreDetailsItems = [],
		birthday = this.person.getBirthday().getDisplayValue(),
		relations = this.person.getRelations().getArray(),
		spouse = _.detect(relations, function (relation) {
			return relation.getType() === Contacts.Relation.TYPE.SPOUSE;
		}),
		child = _.detect(relations, function (relation) {
			return relation.getType() === Contacts.Relation.TYPE.CHILD;
		}),
		nickname = this.person.getNickname().getDisplayValue();
	
	if (birthday) {
		moreDetailsItems.push({
			key: RB.$L("birthday"),
			value: birthday
		});
	}
	if (spouse) {
		moreDetailsItems.push({
			key: RB.$L("spouse"),
			value: spouse.getDisplayValue()
		});
	}
	if (child) {
		moreDetailsItems.push({
			key: RB.$L("children"),
			value: child.getDisplayValue()
		});
	}
	if (nickname) {
		moreDetailsItems.push({
			key: RB.$L("nickname"),
			value: nickname
		});
	}
	
	return moreDetailsItems;
};

DetailWidget.prototype.renderReminder = function () {
	var reminder = this.person.getReminder().getValue();
	if (reminder && reminder.length > 0) {
		this.reminderContainer.innerHTML = Mojo.View.render({
			templateRoot: LIB_ROOT,
			template: Utils.getTemplatePath("detail-widget/reminder"),
			object: {reminder: reminder}
		});
		Mojo.Dom.show(this.reminderContainer);
	} else {
		Mojo.Dom.hide(this.reminderContainer);
	}
};

DetailWidget.prototype.handleReminderTap = function () {
	pushReminderScene(this.controller.stageController, {
		person: this.person,
		fullName: this.person.getName().getFullName(),
		reminderText: this.person.getReminder().getValue() || "",
		focusField: true
	});
};

// This searches the DOM for a div with the class "new-defer" and changes
// the display style to inherit.  This is being used to make the hidden
// lists for contact data visible.  showWidgetContainer is probably called on each
// item to make everything render properly
DetailWidget.prototype.showDeferrals = function () {
	var deferrals, i;
	
	if (this.hasShowDeferralsBeenExecuted) {
		return;
	}
	
	// FIXME: this change this to use an attribute instead of a class
	deferrals = this.controller.select(".new-defer");
	for (i = 0; i < deferrals.length; i += 1) {
		deferrals[i].style.display = "inherit";
		this.controller.showWidgetContainer(deferrals[i]);
	}
	this.hasShowDeferralsBeenExecuted = true;
};
	
DetailWidget.prototype.moreDetailNames = ["birthday", "spouse", "children", "nickname"];

DetailWidget.prototype.toggleLinked = function (event) {
	//Don't open if it's a pseudo card
	if (!this._widget.model.hideLinkedContacts) {
		Mojo.Dom.toggleClassName(this.header, "drawer-open");
		this.headerDrawer.mojo.toggleState();
	}
};

DetailWidget.prototype.onClippedChoose = function (contact, node, selection) {
	var foundPrimary = false,
		future;
	
	if (selection === 'UNCLIP') {
		Mojo.Log.info("manualunlink contact.getId(): " + contact.getId());
		future = Contacts.Person.manualUnlink(this.person.getId(), contact.getId());
		future.then(this, function (future) {
			var result = future.result;
			
			Mojo.Log.info("\n\n\n^^^^^^^manualUnLink RESULT!!!!: " + JSON.stringify(result));
			this.onSynchronousChange();
			
			future.result = true;
		});
	} else if (selection === 'MAKE_PRIMARY') {
		foundPrimary = this.person.setContactWithIdAsPrimary(contact.getId());
		if (foundPrimary) {
			future = this.person.fixup();
			future.then(this, function (future) {
				var result = future.result;
				return this.person.save();
			});
			
			future.then(this, function () {
				var result = future.result;
				return this.onSynchronousChange();
			});
		}
	} else if (selection === 'DELETE_PROFILE') {
		Mojo.Dom.addClassName(node, "deleting");
		future = contact.deleteContact();
		
		future.then(this, function () {
			var dummy = future.result;
			
			console.log("Finished deleting contact. calling renderPersonAndLinkedContacts()");
			this.renderPersonAndLinkedContacts();
			// This will cause the callback to fire now and then again when the DB watch fires
			// we might want to handle this more intelligently
			this.fireRenderAfterUpdateCallback();
			
			future.result = true;
		});
	}
};

DetailWidget.prototype.launchOtherApp = function (appId, params) {
	Mojo.Log.info("Launching app " + appId);
	
	return this.controller.serviceRequest("palm://com.palm.applicationManager", {
		method: 'open',
		parameters: {
			id: appId,
			params: params
		}
	});
};

DetailWidget.prototype.handleListTap = function (e) {
	var eventTarget = this.controller.get(e.originalEvent.target),
		item = e.item,
		useMessaging = false,
		usePhone = false,
		launchParams;
	
	// determine if the SMS icon next to a phone number was tapped
	if (Mojo.Dom.hasClassName(eventTarget, 'messaging')) {
		useMessaging = true;
	}
	
	// determine if the skype icon next to an im address was tapped
	if (Mojo.Dom.hasClassName(eventTarget, 'phone')) {
		usePhone = true;
	}
	
	//if we didn't decide on phone vs. messaging above, we use the type of the item
	if (!useMessaging && !usePhone) {
		if (item instanceof Contacts.IMAddress) {
			useMessaging = true;
		} else if (item instanceof Contacts.PhoneNumber) {
			usePhone = true;
		}
	}
	
	if (useMessaging) {
		launchParams = {
			compose: {
				personId: this.person.getId()
			}
		};
		
		if (item instanceof Contacts.IMAddress) {
			launchParams.compose.ims = [item.getDBObject()];
		} else {
			launchParams.compose.phoneNumbers = [item.getDBObject()];
		}
		
		return this.launchOtherApp("com.palm.app.messaging", launchParams);
	} else if (usePhone) {
		launchParams = {
			personId: this.person.getId(),
			address: item.getValue(),
			addressLabel: item.getType()
		};
		
		if (item instanceof Contacts.IMAddress) {
			launchParams.service = item.getType();
		}
		
		return this.launchOtherApp("com.palm.app.phone", launchParams);
	} else if (item instanceof Contacts.Address) {
		this.openWith.shouldOpenOpenWith(e);
	} else if (item instanceof Contacts.EmailAddress) {
		return this.launchOtherApp("com.palm.app.email", {
			uri: "mailto:" + item.getValue()
		});
	} else if (item instanceof Contacts.Url) {
		return this.launchOtherApp("com.palm.app.browser", {
			scene: 'page',
			url: item.getValue()
		});
	}
};


//@ sourceURL=contacts.ui/PeoplePickerAssistant.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Contacts, RB, Mojo, _, Utils, EventListenerManager, PersonListWidget, LIB_ROOT*/


var PeoplePickerAssistant = function (params) {
	this.eventListenerManager = new EventListenerManager();
	
	if (params) {
		//this.suppressPopScene = params.suppressPopScene;
		
		// By default this scene should be popped when a person is selected.  This scene is typically a temporary selector
		// In the event that tapping on a person should push a scene on top of the people picker, set popSceneOnPersonSelect to false
		this.popSceneOnPersonSelect = (params.popSceneOnPersonSelect !== undefined ? params.popSceneOnPersonSelect : true);
		this.focusWindow = params.focusWindow;
		this.favoritesOnly = params.favoritesOnly;
		this.excludeFavorites = params.excludeFavorites;
		this.personExclusions = params.exclusions;
		this.pickerMessage = params.message;
		this.iconClass = params.iconClass;
		this.returnRawResult = params.returnRawResult;
		this.peoplePickerCallback = params.callback;
	}
};

PeoplePickerAssistant.prototype.setup = function () {	
	var personListModel,
		sortOrder;
	
	this.listDiv = this.controller.get('peoplepicker-person-list');
	this.listHeader = this.controller.get("peoplepicker-header");
	
	if (this.pickerMessage) {
		this.listHeader.innerHTML = Mojo.View.render({
			templateRoot: LIB_ROOT,
			template: Utils.getTemplatePath("peoplepicker/header"),
			object: {
				message: this.pickerMessage,
				iconClass: this.iconClass
			}
		});
	}
	
	this.spinnerModel = {
		spinning: false
	};
	this.controller.setupWidget('list-saving-spinner', {
		spinnerSize: 'large'
	}, this.spinnerModel);
	
	// TODO: Need to query for the sortOrder
	//sortOrder = AppAssistant.appPrefs.get(Contacts.AppPrefs.Pref.listSortOrder);
	sortOrder = Contacts.ListWidget.SortOrder.defaultSortOrder;
	
	this.filterFieldElement = this.controller.get("peoplePickerFilterField");
	this.controller.setupWidget('peoplePickerFilterField', {
		delay: 0
	}, {});
	this.eventListenerManager.addListener(this.filterFieldElement, Mojo.Event.filter, this.handleFilter.bind(this));
	this.eventListenerManager.addListener(this.filterFieldElement, Mojo.Event.filterOpened, this.handleFilterOpened.bind(this));
	this.eventListenerManager.addListener(this.filterFieldElement, Mojo.Event.filterClosed, this.handleFilterClosed.bind(this));
	
	this.personListWidget = new PersonListWidget();
	personListModel = {
		mode: PersonListWidget.MODE.PERSON_LIST,
		defaultView: PersonListWidget.VIEW.PERSON_LIST,
		listTapCallback: this.handleListTap.bind(this),
		exclusions: this.personExclusions || [],
		favoritesOnly: this.favoritesOnly,
		excludeFavorites: this.excludeFavorites,
		sortOrder: sortOrder
	};
	this.personListWidget.setup(this.controller, this.listDiv, personListModel);
	
	// remove the background image added by a parent app
	Mojo.Dom.addClassName(this.controller.document.body, "no-wallpaper");
};

PeoplePickerAssistant.prototype.activate = function () {
	if (this.focusWindow) {
		this.focusWindow = false;
		this.controller.stageController.activate();
	}
};

PeoplePickerAssistant.prototype.deactivate = function () {
	// remove the added classname
	Mojo.Dom.removeClassName(this.controller.document.body, "no-wallpaper");
};

PeoplePickerAssistant.prototype.cleanup = function () {
	this.personListWidget.teardown();
	this.eventListenerManager.destroyListeners();
};

PeoplePickerAssistant.prototype.handleFilterOpened = function (event) {
	Mojo.Dom.hide(this.listHeader);
};

PeoplePickerAssistant.prototype.handleFilterClosed = function (event) {
	Mojo.Dom.show(this.listHeader);
};

PeoplePickerAssistant.prototype.handleFilter = function (event) {
	this.personListWidget.filter(event.filterString, this.filterFieldElement.mojo.setCount);
};

PeoplePickerAssistant.prototype.handleSortOrder = function (response) {
	this.sortOrder = response.order;
	
	// FIXME: I don't think this works.  Can a list have its attributes dynamically like this?  --andy
	//this.listAttrs.dividerTemplate = (this.sortOrder === PrefsConstants.companyFirstLast || this.sortOrder === PrefsConstants.companyLastFirst) ? 'list/multiline-separator' : 'list/group-separator';
};

PeoplePickerAssistant.prototype.handleCommand = function (event) {
};

PeoplePickerAssistant.prototype.handleListTap = function (event) {
	var personLite = event.item,
		person;
	
	Mojo.Log.info("tapped on " + event.item.display + " at time=" + Date.now());
	if (this.returnRawResult) {
		person = personLite;
	} else {
		person = Contacts.PersonFactory.createPersonDisplay(personLite);
	}
	
	if (_.isFunction(this.peoplePickerCallback)) {
		if (this.popSceneOnPersonSelect) {
			this.controller.stageController.popScene();
		}
		this.peoplePickerCallback(person);
	} else {
		// JAHACK
		//AppAssistant.contactsService.details(returnParams.personId, this.peoplePickerDetailCallback.bind(this, returnParams), this.controller, null, this.peoplePickerDetailCallbackFailure.bind(this));
	}
};

//peoplePickerDetailCallbackFailure: function (resp) {
//	this.controller.stageController.popScene(resp);
//},
//
//peoplePickerDetailCallback: function (oldParams, resp) {
//	oldParams.details = resp;
//	this.controller.stageController.popScene(oldParams);
//},

// called from the app-assistant via delegateToSceneAssistant
PeoplePickerAssistant.prototype.showListSavingScrim = function () {
	Mojo.Dom.show(this.controller.get("list-saving-div"));
	this.spinnerModel.spinning = true;
	this.controller.modelChanged(this.spinnerModel);
};

PeoplePickerAssistant.prototype.showErrorMessage = function (error) {
	this.controller.showAlertDialog({
		message: error,
		choices: [{
			label: RB.$L("OK"),
			command: "ok"
		}],
		onChoose: function () {}
	});
};


//@ sourceURL=contacts.ui/PseudoDetailAssistant.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Contacts, Mojo, _, Utils, RB, ContactsUI, DetailWidget, Assert, Future */

var PseudoDetailAssistant = function (params, contactsFuture) {
	this.params = params || {};
	this.contactsFuture = contactsFuture;
	this.mode = PseudoDetailAssistant.MODE.EDIT;
	this.focusWindow = this.params.focusWindow;
	
	if (this.params.contact) {
		Assert.require(this.params.contact instanceof Contacts.Contact, "PseudoDetailAssistant requires a decorated Contact");
	}
	
	if (this.params.person) {
		Assert.require(this.params.person instanceof Contacts.Person, "PseudoDetailAssistant requires a decorated Person");
	}
	
	Assert.require(this.params.contact || this.params.person || this.params.personId, "PseudoDetailAssistant requires a decorated Person, a decorated Contact, or a person id");
	
	this.handleStageBlur = this.handleStageBlur.bind(this);
};
	
PseudoDetailAssistant.prototype.setup = function () {
	var future = new Future();
	
	future.now(this, function () {
		this.detailWidget = new DetailWidget();
		
		// if a contact has been specified then we are in ADD mode... convert the contact into a person
		// for display only. Keep a reference to the original contact object since this will be what we end up saving
		this.contact = this.params.contact;
		var detailModel = {
				person: (this.contact ? Contacts.PersonFactory.createPersonDisplay(this.contact) : this.params.person),
				personId: this.params.personId,
				contactsFuture: this.contactsFuture
			},
			commandMenuItemModel = {
				visible: true,
				items: []
			};
		
		// If a contact was passed, then we are in ADD mode
		if (this.contact) {
			detailModel.hideLinkedContacts = true;
			this.mode = PseudoDetailAssistant.MODE.ADD;
		}
		
		// setup the command menu
		if (this.mode === PseudoDetailAssistant.MODE.EDIT) {
			commandMenuItemModel.items.push({
				label: RB.$L('Edit'),
				command: "edit"
			});
		} else {
			commandMenuItemModel.items.push({
				label: RB.$L('Add To Contacts'),
				command: "add"
			});
		}
		this.controller.setupWidget(Mojo.Menu.commandMenu, undefined, commandMenuItemModel);
		
		return this.detailWidget.setup(this.controller, this.controller.get("pseudodetail"), detailModel);
	});
	
	return future;
};

PseudoDetailAssistant.prototype.activate = function () {
	if (this.focusWindow) {
		this.focusWindow = false;
		this.controller.stageController.activate();
	}
	this.controller.listen(this.controller.document, Mojo.Event.stageDeactivate, this.handleStageBlur);
};

PseudoDetailAssistant.prototype.deactivate = function () {
	this.controller.stopListening(this.controller.document, Mojo.Event.stageDeactivate, this.handleStageBlur);
};

PseudoDetailAssistant.prototype.cleanup = function () {
	this.detailWidget.teardown();
};

PseudoDetailAssistant.prototype.handleCommand = function (event) {
	if (event.type === Mojo.Event.command) {
		if (event.command === 'edit') {
			this.launchContactsAppToEditView();
		} else if (event.command === 'add') {
			this.addToContactsPopup();
		}
	}
};

PseudoDetailAssistant.prototype.handleStageBlur = function () {
	this.controller.stageController.popScene();
};

PseudoDetailAssistant.prototype.launchContactsAppToEditView = function () {
	this.launchContactsApp({
		launchType: "editContact",
		person: this.detailWidget.getPerson().getDBObject()
	});
};

PseudoDetailAssistant.prototype.addToContactsPopup = function () {
	this.controller.showAlertDialog({
		onChoose: this.addToContactsPopupChoice.bind(this),
		choices: [{
			label: RB.$L('Add New Contact'),
			value: 'NEW'
		}, {
			label: RB.$L('Add to existing'),
			value: 'EXISTING'
		}, {
			label: RB.$L('Cancel'),
			value: 'CANCEL'
		}]
	});
};

PseudoDetailAssistant.prototype.addToContactsPopupChoice = function (choice) {
	if (choice === 'CANCEL') {
		return;
	} else if (choice === 'NEW') {
		this.launchContactsApp({
			contact: this.contact.getDBObject(),
			launchType: "newContact"
		});
	} else if (choice === 'EXISTING') {
		this.launchContactsApp({
			person: this.detailWidget.getPerson().getDBObject(),
			launchType: "addToExisting"
		});
	}
};

PseudoDetailAssistant.prototype.launchContactsApp = function (params) {
	this.controller.serviceRequest("palm://com.palm.applicationManager", {
		method: "launch",
		parameters: {
			id: "com.palm.app.contacts",
			params: params
		}
	});
	this.controller.stageController.popScene();
};

PseudoDetailAssistant.MODE = {
	EDIT: "mode_edit",
	ADD: "mode_add"
};


//@ sourceURL=contacts.ui/ReminderAssistant.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, AppAssistant, Contacts, Future, Utils, RB, Assert, LIB_ROOT */

var ReminderAssistant = function (params) {
	this.focusField = params.focusField;
	this.focusWindow = params.focusWindow;
	this.person = params.person;
	this.personId = params.personId;
	this.reminderText = params.reminderText;
	this.fullName = params.fullName;
	this.hasReminderTextToLoad = params.hasReminderTextToLoad;
	this.deleteReminder = false;
	
	// keep a reference to the future where the person object is retrieved.  This
	// will be used in the event that the user enters/exits this scene before the person
	// has been retrieved.
	this.fetchPersonFuture = new Future();	
	
	Assert.require((this.person && this.person instanceof Contacts.Person) || this.personId, "ReminderAssistant requires that you pass a decorated person or personId");
};

ReminderAssistant.prototype.setup = function () {
	var hintTextToDisplay;
	
	this.fetchPersonFuture.now(this, function () {
		if (this.person && this.person instanceof Contacts.Person) {
			this.fetchPersonFuture.result = this.person;
		} else {
			this.fetchPersonFuture.nest(Contacts.Person.findById(this.personId));
		}
	});
	
	this.reminderModel = {
		reminder: this.reminderText
	};
	
	hintTextToDisplay = ReminderAssistant.reminderTextFieldHintValues.create;
	
	if (this.hasReminderTextToLoad) {
		hintTextToDisplay = ReminderAssistant.reminderTextFieldHintValues.loading;
	}
	
	this.reminderAttributes = {
		multiline: true,
		modelProperty: "reminder",
		hintText: hintTextToDisplay
	};
	this.controller.setupWidget("reminderTextField", this.reminderAttributes, this.reminderModel);
	
	this.controller.setupWidget(Mojo.Menu.commandMenu, undefined, {
		items: [{
			icon: 'delete',
			label: 'Clear',
			command: 'clear'
		}]
	});
	
	if (!this.focusField) {
		this.controller.setInitialFocusedElement(null);
	}
	
	this.fetchPersonFuture.then(this, function () {
		this.person = this.fetchPersonFuture.result;
		this.renderHeader(this.person.getName().getFullName());
		this.reminderModel.reminder = this.person.getReminder().getValue();
		this.reminderAttributes.hintText = ReminderAssistant.reminderTextFieldHintValues.create;
		this.controller.modelChanged(this.reminderModel);
		this.fetchPersonFuture.result = this.person; // set the result in case this future is needed in cleanup() when we save
	});
};

ReminderAssistant.prototype.renderHeader = function (fullName) {
	var header = Mojo.View.render({
			templateRoot: LIB_ROOT,
			template: Utils.getTemplatePath("reminder/header"),
			object: {fullName: fullName}
		});
	
	this.controller.get("reminderHeader").innerHTML = header;
};

ReminderAssistant.prototype.handleCommand = function (event) {
	if (event.type === Mojo.Event.command && event.command === 'clear') {
		this.clearReminder();
	}
};

ReminderAssistant.prototype.serviceFailed = function () {
	this.controller.stageController.popScene();
};

ReminderAssistant.prototype.activate = function () {
	if (this.focusWindow) {
		this.controller.stageController.activate();
		this.focusWindow = false;
	}
};

ReminderAssistant.prototype.clearReminder = function () {
	this.deleteReminder = true;
	this.reminderModel.reminder = null;
	this.controller.stageController.popScene();
};

ReminderAssistant.prototype.cleanup = function () {
	//check to make sure there's a value for this.person just in case the future that fetches the person hasn't finished
	if (this.person) {
		this.person.getReminder().setValue((this.deleteReminder ? "" : this.reminderModel.reminder.trim()));
		this.person.save();
	}
};

ReminderAssistant.reminderTextFieldHintValues = {
	"loading": RB.$L("Loading reminder text..."),
	"create": RB.$L("Create new reminder...")
};


//@ sourceURL=contacts.ui/ContactPointPickerAssistant.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, RB, Contacts, Utils, Assert, _, console, LIB_ROOT*/

var ContactPointPickerAssistant = function (params) {
	Assert.require(params.type, "ContactPointPickerAssistant requires a type");
	
	// Type can be a string or an array of types
	if (_.isString(params.type)) {
		this.types = [params.type];
	} else if (_.isArray(params.type)) {
		this.types = params.type;
	}		
	Assert.requireArray(this.types, "ContactPointPickerAssistant invalid type param");

	this.person = params.person;
	this.personId = params.personId;
	this.message = params.message;
	this.callback = params.callback;
	this.popSceneOnItemSelect = (params.popSceneOnItemSelect !== undefined ? params.popSceneOnItemSelect : true);
	this.showSpeedDialButtons = params.showSpeedDialButtons;
	this.saveSelectionAsPrimary = params.saveSelectionAsPrimary;
	this.uniqueId = "CPP_" + Date.now();
	this.stylesheets = [Utils.getStylesheetPath("contactpointpicker.css")];
	this.listIndexForDefault = params.listIndexForDefault;
	this.auxFavoriteDataForDefault = params.auxFavoriteDataForDefault;
};

ContactPointPickerAssistant.prototype.setup = function () {
	Utils.loadSceneStyles(this.controller, this.stylesheets, this.uniqueId);
	
	// remove the background image added by a parent app
	Mojo.Dom.addClassName(this.controller.document.body, "no-wallpaper");
	
	this.listAttrs = {
		//emptyTemplate: Utils.getTemplatePath("contactpointpicker/empty"),
		itemTemplate: Utils.getTemplatePath("contactpointpicker/item"),
		templateRoot: LIB_ROOT
	};
	this.listModel = {
		items: []
	};
	this.controller.setupWidget("contactpoint-picker-list", this.listAttrs, this.listModel);
	this.controller.listen("contactpoint-picker-list", Mojo.Event.listTap, this.handleItemTap.bind(this));
	
	if (this.person) {
		this.renderHeaderAndContactPoints();
	} else if (this.personId) {
		// fetch the person because we currently only have the id
		Contacts.Person.findById(this.personId).then(this, function (future) {
			this.person = future.result;
			this.renderHeaderAndContactPoints();
			future.result = true;
		});
	}
};

ContactPointPickerAssistant.prototype.deactivate = function () {
	// remove the added classname
	Mojo.Dom.removeClassName(this.controller.document.body, "no-wallpaper");
};

ContactPointPickerAssistant.prototype.cleanup = function () {
	this.controller.stopListening("contactpoint-picker-list", Mojo.Event.listTap, this.handleItemTap.bind(this));
	Utils.unloadSceneStyles(this.controller, this.stylesheets, this.uniqueId);
};

ContactPointPickerAssistant.prototype.renderHeaderAndContactPoints = function () {
	var i,
		type,
		items = [];
	
	this.controller.get("contactpoint-picker-header").innerHTML = Mojo.View.render({
		templateRoot: LIB_ROOT,
		template: Utils.getTemplatePath("contactpointpicker/header"),
		object: {
			displayName: this.person.generateDisplayName(),
			message: this.message
		}
	});
	
	for (i = 0; i < this.types.length; i += 1) {
		type = this.types[i].contactPoint ? this.types[i].contactPoint : this.types[i];
		
		switch (type) {
		case Contacts.ContactPointTypes.PhoneNumber:
			items.push.apply(items, Utils.getCPsForPersonForType(this.person, Contacts.ContactPointTypes.PhoneNumber, this.types[i].type));
			break;

		case Contacts.ContactPointTypes.EmailAddress:
			items.push.apply(items, Utils.getCPsForPersonForType(this.person, Contacts.ContactPointTypes.EmailAddress, this.types[i].type));
			break;
			
		case Contacts.ContactPointTypes.IMAddress:
			items.push.apply(items, Utils.getCPsForPersonForType(this.person, Contacts.ContactPointTypes.IMAddress, this.types[i].type));
			break;
			
		default:
			console.warn("Encountered an unsupported type: " + type);
			break;
			
		}
	}
	
	this.decorateItems(items);
	
	this.listModel.items = items;
	this.controller.modelChanged(this.listModel);
};

ContactPointPickerAssistant.prototype.decorateItems = function (items) {
	var that = this;
	
	if (!items || !_.isArray(items)) {
		items = [];
	}
	
	items.forEach(function (item) {
		if (item.getPrimary() || item.getFavoriteDataForAppWithId(Mojo.appInfo.id)) {
			item.primaryClass = ContactPointPickerAssistant.PRIMARY_CLASS;
		}
		
		if (that.showSpeedDialButtons && _.isFunction(item.getSpeedDial) && item.getSpeedDial()) {
			item.hasSpeedDial = "show-quick-dial";
		}
	});
	
	return items;
};

ContactPointPickerAssistant.prototype.handleItemTap = function (e) {
	var item = e.item;
	
	if (this.saveSelectionAsPrimary) {
		ContactPointPickerAssistant.setItemAsDefault(item, this.person, this.listIndexForDefault, this.auxFavoriteDataForDefault, this.callback);
	} else {
		ContactPointPickerAssistant.callCallback(this.callback, item, this.person);
	}
		
	if (this.popSceneOnItemSelect) {
		this.controller.stageController.popScene();
	}

};

ContactPointPickerAssistant.callCallback = function (callback, item, person) {
	if (callback && _.isFunction(callback)) {
		callback(item, person);
	}
};

ContactPointPickerAssistant.setItemAsDefault = function (item, person, listIndex, auxFavoriteDataForDefault, callback) {
	var param = { 
			defaultData: { 
				value: item.getValue(), 
				listIndex: listIndex,
				auxData: auxFavoriteDataForDefault
			} 
		};
	
	if (item instanceof Contacts.PhoneNumber) {
		param.defaultData.contactPointType = Contacts.ContactPointTypes.PhoneNumber;
	} else if (item instanceof Contacts.IMAddress) {
		param.defaultData.contactPointType = Contacts.ContactPointTypes.IMAddress;
	} else if (item instanceof Contacts.EmailAddress) {
		param.defaultData.contactPointType = Contacts.ContactPointTypes.EmailAddress;
	} 
	
	if (param.defaultData.contactPointType) {
		ContactPointPickerAssistant.callCallback(callback, item, person);
		person.setFavoriteDefault(param);
	} else {
		ContactPointPickerAssistant.callCallback(callback, item, person);
	}
};



// This will be set as a property on list items that have primary set to true
// This will be rendered into the item template as a CSS class
ContactPointPickerAssistant.PRIMARY_CLASS = "primary";


//@ sourceURL=contacts.ui/FavoritePersonWidget.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, Contacts, Utils, Assert, _, console, exports, pushPeoplePickerScene, pushContactPointPickerScene, ContactPointPickerAssistant */

var FavoritePersonWidget = exports.FavoritePersonWidget = function (params) {
	Assert.require(params.defaultType, "FavoritePersonWidget requires a contact point default type");
	
	// Type can be a string or an array of types
	if (_.isString(params.defaultType)) {
		this.defaultType = [params.defaultType];
	} else if (_.isArray(params.defaultType)) {
		this.defaultType = params.defaultType;
	}		
	Assert.requireArray(this.defaultType, "FavoritePersonWidget invalid defaultType param");
	
	this.hashedDefaultType = FavoritePersonWidget._transformDefaultTypesArrayIntoHash(this.defaultType);

	this.scenesToPush = FavoritePersonWidget._transformArrayIntoHash(params.scenesToPush || [FavoritePersonWidget.SCENES.CONTACT_POINT_PICKER]);
	this.stageController = params.stageController;
	Assert.require(this.stageController, "FavoritePersonWidget you must specify the stageController");
	this.personPickerMessage = params.personPickerMessage || "";
	this.contactPointPickerMessage = params.contactPointPickerMessage || "";
	Assert.require(this.scenesToPush[FavoritePersonWidget.SCENES.PERSON_PICKER] || (params.person || params.personId), "FavoritePersonWidget you must provide either a person or personId param or add the person picker to the scenes to push");
	this.person = params.person;
	
	if (this.person) {
		Assert.require(this.person instanceof Contacts.Person, "FavoritePersonWidget person param must be an instance of Person");
	}
	
	this.personId = params.personId;
	this.callback = params.callback;
	Assert.require(this.callback, "FavoritePersonWidget you must specify a callback function");
	this.auxFavoriteDataForDefault = params.auxFavoriteDataForDefault;
	this.listIndexForDefault = params.listIndexForDefault;
	
	if (!this.listIndexForDefault && this.listIndexForDefault !== 0) {
		this.listIndexForDefault = -1;
	}
	
};

FavoritePersonWidget.prototype.pushScene = function () {
	var param;
	if (this.scenesToPush[FavoritePersonWidget.SCENES.PERSON_PICKER]) {
		param = {
			excludeFavorites: true,
			iconClass: "icon-add-fav",
			message: this.personPickerMessage,
			popSceneOnPersonSelect: false,
			callback: this._tappedOnPersonToFavorite.bind(this)
		};
		
		pushPeoplePickerScene(this.stageController, param);
	} else {
		param = {
			type: this.defaultType,
			message: this.contactPointPickerMessage,
			saveSelectionAsPrimary: true,
			listIndexForDefault: this.listIndexForDefault,
			auxFavoriteDataForDefault: this.auxFavoriteDataForDefault,
			callback: this.callback
		};
		
		if (this.person) {
			param.person = this.person;
		} else {
			param.personId = this.personId;
		}
		
		pushContactPointPickerScene(this.stageController, param);
	}
	
};

// Determines if the person has enough contact points to push the contact point
// picker. If there is only one contact point of the type then just return that
// item. If there are no contact points of the type then return false. Otherwise,
// just return true.
// TODO: make this _shouldPushCPPickerOrGetDefaultItem
FavoritePersonWidget.prototype.shouldPushCPPickerOrGetDefaultItem = function () {
	var contactPoints = [],
		that = this;
	
	// Done: add some comments so other beings can understand the immense quantity of density these 3 lines have (This comment was left because of it's pure awesomeness when I came across it)
	// Iterate over all the different kinds of contact points. Check to see if they are in the list of types that should be displayed.
	// Get all the contact points of that type from the person and add them to the contactPoints array.
	Object.keys(Contacts.ContactPointTypes).forEach(function (key) {
		if (that.hashedDefaultType[Contacts.ContactPointTypes[key]]) {
			contactPoints = contactPoints.concat(Utils.getCPsForPersonForType(that.person, Contacts.ContactPointTypes[key], that.hashedDefaultType[Contacts.ContactPointTypes[key]].type));
		}
	});
	
	if (contactPoints.length > 1) {
		return true;
	} else if (contactPoints.length === 1) {
		return contactPoints[0];
	} else {
		return false;
	}
};

FavoritePersonWidget.prototype._tappedOnPersonToFavorite = function (person) {
	this.person = person;
	this.person.makeFavorite(true).then(this, function (future) {
		var result = future.result;
		
		if (!result) {
			throw new Error("Making person favorite failed!");
		}
	});
	
	var shouldPushContactPointPickerOrDefaultItem = this.shouldPushCPPickerOrGetDefaultItem(),
		isRetValABoolean = typeof(shouldPushContactPointPickerOrDefaultItem) === "boolean";
	if (isRetValABoolean && shouldPushContactPointPickerOrDefaultItem) {
		pushContactPointPickerScene(this.stageController, {
			person: this.person,
			type: this.defaultType,
			message: this.contactPointPickerMessage,
			saveSelectionAsPrimary: true,
			popSceneOnItemSelect: false,
			listIndexForDefault: this.listIndexForDefault,
			auxFavoriteDataForDefault: this.auxFavoriteDataForDefault,
			callback: function (item, person) {
				this.stageController.popScene();
				this.stageController.popScene();
				this.callback(item, person);
			}.bind(this)
		});
	} else if (isRetValABoolean) {
		this.callback(undefined, this.person);
	} else if (shouldPushContactPointPickerOrDefaultItem) {
		ContactPointPickerAssistant.setItemAsDefault(shouldPushContactPointPickerOrDefaultItem, this.person, this.listIndexForDefault, this.auxFavoriteDataForDefault, this.callback);
	}
};

FavoritePersonWidget._transformDefaultTypesArrayIntoHash = function (array) {
	var toReturn = {};
	
	array.forEach(function (item) {
		if (_.isString(item)) {
			toReturn[item] = { contactPoint: item };
		} else if (item.contactPoint && _.isString(item.contactPoint)) {
			toReturn[item.contactPoint] = item;
		}
	});
	
	return toReturn;
};

FavoritePersonWidget._transformArrayIntoHash = function (array) {
	var toReturn = {};
	
	array.forEach(function (item) {
		if (_.isString(item)) {
			toReturn[item] = true;
		}
	});
	
	return toReturn;
};

FavoritePersonWidget.SCENES = {
	PERSON_PICKER: "favorite_person_picker",
	CONTACT_POINT_PICKER: "favorite_contact_point_picker"
};

//@ sourceURL=contacts.ui/SimloginAssistant.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, Contacts, RB, Assert, LIB_ROOT, exports, EventListenerManager, Utils, console, Globalization */

var SimloginAssistant = exports.SimloginAssistant = function (params) {
	this.attrs = params.attrs;
	this.model = params.model;
	//unique identifier for the stylesheet in order to custom load it
	//only for this scene and unload it after the scene is popped
	//else it will unload the global stylesheet with same name for the other scenes also
	this.uniqueId = "SLA_" + Date.now(); 
	this.eventListenerManager = new EventListenerManager();

	this.stylesheets = [Utils.getStylesheetPath("common.css")];
};

SimloginAssistant.prototype.setup = function () {
	var simSection, userName, phoneNumber;

	Utils.loadSceneStyles(this.controller, this.stylesheets, this.uniqueId);

	this.controller.get('phoneIdLabel').innerText = this.attrs.phoneIdLabel;
	this.controller.get('simTitle').innerText = this.attrs.simTitle; 
	this.controller.get('simText').innerText = this.attrs.simText; 

	phoneNumber = Globalization.Format.formatPhoneNumber(Globalization.Phone.parsePhoneNumber(this.model.phoneId));
	
	userName = {
		value: phoneNumber,
		disabled: true
	};
	this.controller.setupWidget('phoneId', {}, userName);

	try {
		this.controller.get('LoginHeaderDiv').innerHTML = Mojo.View.render({
			object: {
				iconPath: this.model.account.icon.loc_32x32
			},
			template: Utils.getTemplatePath('simlogin/simlogin-header'),
			templateRoot: LIB_ROOT
		});
	} catch ( e ) {
		console.log("Exception caught: " + e);
		console.log(e.stack());
	}
	
	simSection = this.controller.get('sim');
	if (simSection && this.model.hideSimRemovedMessage) {
		Mojo.Dom.hide(simSection);
	}
	
	// remove account button
	this.removeAccountAttributes = {
		type: Mojo.Widget.activityButton
	};
	this.removeAccountModel = {
		label : RB.$L('Remove Account'),
		disabled: false,
		buttonClass: "negative"
	};

	this.controller.setupWidget('removeAccount-button', this.removeAccountAttributes, this.removeAccountModel);
	this.removeAccountButton = this.controller.get('removeAccount-button');
	if (this.model.hideSimRemovedMessage) {
		Mojo.Dom.hide(this.removeAccountButton);
	} else {
		Mojo.Dom.show(this.removeAccountButton);
		this.eventListenerManager.addListener(this.removeAccountButton, Mojo.Event.tap, this.handleRemoveDialog.bind(this));
	}
	
	Assert.require(this.model.fetchContactCount && typeof this.model.fetchContactCount === 'function', "SimloginAssistant.setup: fetchContactCount callback is not function");
	this.fetchFuture = this.model.fetchContactCount(this.model.account._id, this.controller.get('PersonCountBlock'), this.controller.get('NumberOfPeople'));
};

SimloginAssistant.prototype.cleanup = function () {
	Utils.unloadSceneStyles(this.controller, this.stylesheets, this.uniqueId);
	this.eventListenerManager.destroyListeners();
};

SimloginAssistant.prototype.handleRemoveDialog = function () {
	var that = this;
	
	//console.log("SimloginAssistant.handleRemoveDialog called");
	
	this.controller.showAlertDialog({
		onChoose: function (value) {
			var future;
			
			if (value === "remove") {
				that.removeAccountButton.mojo.activate();
				that.removeAccountButton.buttonLabel = RB.$L("Removing Account...");
				that.removeAccountButton.disabled = true;
				
				Assert.require(that.model.handleRemoveAccount && typeof that.model.handleRemoveAccount === 'function', "SimloginAssistant.handleRemoveDialog: handleRemoveAccount callback is not function");
				future = that.model.handleRemoveAccount(that.model.account);
				
				future.then(that, function () {
					that.removeAccountButton.mojo.deactivate();
					Assert.require(that.model.endFetchContactCount && typeof that.model.endFetchContactCount === 'function', "SimloginAssistant.handleRemoveDialog: endFetchContactCount callback is not function");
					that.model.endFetchContactCount();
					that.controller.stageController.popScene();
				});
			} else {
				that.removeAccountButton.buttonLabel = RB.$L("Remove Account");
				that.removeAccountButton.mojo.deactivate();
				that.removeAccountButton.disabled = false;
				return;
			}
		},
		title: RB.$L("Remove Account"),
		message: RB.$L("Are you sure you want to remove this Contact account, and all linked data, from this device?"),
		preventCancel: true,
		choices: [{
			label: RB.$L("Remove contact account"),
			value: "remove",
			type: 'negative'
		}, {
			label: RB.$L("Keep contact account"),
			value: "keep"
		}]
	});
};


//@ sourceURL=contacts.ui/Utilities/AccountList.js

/* Copyright 2010 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Assert, Mojo, _, console, Contacts, Utils, AccountsLib */

// A simple class to encapsulate the global accounts list for the email app.
// Extends the accounts.ui library support by adding an account ID lookup table, a facility for "change listeners", and coordinated functionality for com.palm.mail.account objects.
// onReady function will be called after the accounts list has been loaded.
var AccountList = exports.AccountList = function (onReady) {
	this.list = [];
	this.ids = {}; // hash of accounts by accountId
	this.providersBySubKinds = {}; // TODO: Possibly take this out when exchange fixes their account stuff
	this.templateIds = {};
	this._fetchingAccounts = false;
	
	this._onReady = onReady;
	
	this.refresh();
	
	Contacts.Utils.mixInBroadcaster(this);
	this._defaultAccountsListeners = [];
};

//copied from mixInBroadcaster
AccountList.prototype.addDefaultAccountsListener = function (callback) {
	if (callback) {
		this._defaultAccountsListeners.push(callback);
	}
};

//copied from mixInBroadcaster
AccountList.prototype._notifyDefaultAccountsListeners = function (newDefaultAccountsList, oldDefaultAccountsList) {
	// Call all listeners with whatever arguments we were passed.
	_.invoke(this._defaultAccountsListeners, "apply", undefined, arguments);
};

//copied from mixInBroadcaster
AccountList.prototype.removeDefaultAccountsListener = function (callback) {
	var i = this._defaultAccountsListeners.indexOf(callback);
	if (i !== -1) {
		this._defaultAccountsListeners.splice(i, 1);
	} else {
		console.error("removeDefaultAccountsListener: Cannot find callback to remove.");
	}
};

AccountList.prototype.refresh = function () {
	this._fetchingAccounts = true;
	
	if (this._listObj) {
		this._listObj.cancel();
	}
	
	//TODO: do we need to get IM accounts too, for their contacts?
	this._listObj = AccountsLib.AccountsList.listAccounts(this._accountsChanged.bind(this), {
//		filterBy: {
//			capability: 'CONTACTS'
//		}, 
		subscribe: true
	});
};

AccountList.prototype.getIconById = function (accountId, big) {
	var provider,
		account,
		iconData;
	
	// Get the contacts capability provider info for this account
	provider = this.getProvider(accountId);
	// Return one icon or the other, depending on whether or not the "big" one was requested.
	if (provider && provider.icon) {
		iconData = provider.icon;
	} else {
		// There is no capability specific icon so get the account's icon
		account = this.getAccount(accountId);
		if (account && account.icon) {
			iconData = account.icon;
		}
	}
	
	if (iconData) {
		if (big) {
			return iconData.loc_48x48;
		} else {
			return iconData.loc_32x32;
		}
	}
};

AccountList.prototype.getIconByKind = function (kindName, big) {
	var provider = this.getProviderByKind(kindName);
	
	if (provider && provider.icon) {
		if (big) {
			return provider.icon.loc_48x48;
		} else {
			return provider.icon.loc_32x32;
		}
	}
};

/** 
 * This method supplies an icon for an account
 */
AccountList.prototype.getAccountIcon = function (accountId, kindName) {
	var icon;
	
	if (kindName) {
		icon = this.getIconByKind(kindName, false);
	}
	
	if (!icon && accountId) {
		icon = this.getIconById(accountId, false);
	}
	
	return icon || 'images/header-icon-contacts.png'; 
};

AccountList.prototype.getDisplayName = function (account) {
	if (account.alias) { 
		return account.alias;
	} 
	return account.username;
};  

AccountList.prototype.getAccountName = function (accountId) {
	var account = this.getAccount(accountId);
	
	return account ? account.loc_name : "";
};

// Checks if the provider of the contact is read-only or if the contact is from an SDN entry on a SIM
AccountList.prototype.isContactReadOnly = function (contact) {
	var isReadOnly = false,
		provider = this.getProvider(contact.getAccountId().getValue());
	
	if (provider && provider.readOnlyData) 
	{
		isReadOnly = true;
	}
	else if (contact && contact.getDBObject() && contact.getDBObject().simEntryType === "sdn")
	{
		// even if we have a provider that is writable, this particular contact may still be read only ("sdn" on a sim)
		isReadOnly = true;
	}
	return isReadOnly;
};

AccountList.prototype.getAccount = function (accountId) {
	return this.ids[accountId];
};

AccountList.prototype.getAccountsByTemplateId = function (templateId) {
	return this.templateIds[templateId];
};

AccountList.prototype.getProvider = function (accountId) {
	var account = this.ids[accountId];
	
	// This happens when we are called with a fake account ID (like the one used for the unified account dashboard).
	if (!account || !account.capabilityProviders) {
		return;
	}

	// Get the contact capability provider info for this account
	return _.detect(account.capabilityProviders, function (p) {
		return p.capability === "CONTACTS";
	});
};

AccountList.prototype.getProviderByKind = function (kindName) {
	return this.providersBySubKinds[kindName];
};

AccountList.prototype.getAccountsList = function () {
	return this.list;
};

AccountList.prototype.getDefaultAccountsList = function () {
	return this.defaultAccountsList;
};

AccountList.prototype.getDefaultAccountsDisplayList = function () {
	//TODO: should this be owned by the prefs scene and have that scene generate/update the list as appropriate?
	return this.defaultAccountsDisplayList;
};

/*
// An example of what is passed to _accountsChanged:
[{
	"_id": "2AZd",
	"_kind": "com.palm.account:1",
	"_rev": 47405,
	"beingDeleted": false,
	"capabilityProviders": [{
		"_id": "2AZi",
		"capability": "CONTACTS",
		"id": "com.palm.google.contacts",
		"loc_name": "Google Contacts",
		"implementation": "palm://com.palm.service.contacts.google/",
		"onCreate": "palm://com.palm.service.contacts.google/onCreate",
		"onEnabled": "palm://com.palm.service.contacts.google/onEnabled",
		"onDelete": "palm://com.palm.service.contacts.google/onDelete",
		"sync": "palm://com.palm.service.contacts.google/sync"
	}],
	"templateId": "com.palm.google",
	"username": "palmtest4",
	"loc_name": "Google",
	"icon": {
		"loc_32x32": "/usr/palm/public/accounts/com.palm.google/images/google-32x32.png",
		"loc_48x48": "/usr/palm/public/accounts/com.palm.google/images/google-48x48.png"
	},
	"validator": "palm://com.palm.service.contacts.google/checkCredentials"
};
{
	"_id": "2AeJ",
	"_kind": "com.palm.account:1",
	"_rev": 47768,
	"beingDeleted": false,
	"capabilityProviders": [{
		"_id": "2AeO",
		"capability": "CONTACTS",
		"id": "com.palm.yahoo.contacts",
		"loc_name": "Yahoo! Contacts",
		"implementation": "palm://com.palm.service.contacts.yahoo/",
		"onCreate": "palm://com.palm.service.contacts.yahoo/onCreate",
		"onEnabled": "palm://com.palm.service.contacts.yahoo/onEnabled",
		"onDelete": "palm://com.palm.service.contacts.yahoo/onDelete",
		"sync": "palm://com.palm.service.contacts.yahoo/sync",
		"readOnlyData": true
	}],
	"templateId": "com.palm.yahoo",
	"username": "palmtest4",
	"loc_name": "Yahoo!",
	"icon": {
		"loc_32x32": "/usr/palm/public/accounts/com.palm.yahoo/images/yahoo-32x32.png",
		"loc_48x48": "/usr/palm/public/accounts/com.palm.yahoo/images/yahoo-48x48.png"
	},
	"validator": "palm://com.palm.service.contacts.yahoo/checkCredentials"
}]
*/
AccountList.prototype._accountsChanged = function (accounts) {
	var that = this,
		oldDefaultAccountsList = this.defaultAccountsList;
	
	this.list = accounts;
	Mojo.Log.info("ContactsApp._accountsChanged: %d accounts.", accounts.length);
	
	// Build the new ids hash.
	this.ids = {};
	this.templateIds = {};
	
	this.defaultAccountsList = [];
	this.defaultAccountsDisplayList = [];
	
	accounts.forEach(function (account) {
		var id = account._id,
			templateId = account.templateId;
		
		that.ids[id] = account;
		
		if (that.templateIds[templateId]) {
			that.templateIds[templateId].push(account);
		} else {
			that.templateIds[templateId] = [account];
		}
		
		account.capabilityProviders.forEach(function (capabilityProvider) {
			var accountDisplayName;
			
			if (capabilityProvider && capabilityProvider.dbkinds && capabilityProvider.dbkinds.contact) {
				that.providersBySubKinds[capabilityProvider.dbkinds.contact] = capabilityProvider;
			}
			// Create the "defaultAccountsList" for selecting accounts
			if (capabilityProvider && capabilityProvider.capability === "CONTACTS" && !capabilityProvider.readOnlyData && account.templateId !== "com.palm.sim") {
				//console.log(templateId + "is NOT read only!!");
				that.defaultAccountsList.push(account);
				
				accountDisplayName = account.alias || capabilityProvider.loc_name || account.loc_name;
				that.defaultAccountsDisplayList.push({
					label: (account.username) ? accountDisplayName + " (" + account.username + ")" : accountDisplayName,
					secondaryIconPath: (capabilityProvider.icon) ? capabilityProvider.icon.loc_32x32 : account.icon.loc_32x32,
					command: account._id
				});
			}
		});
	});
	
	this._fetchingAccounts = false;
	// Notify listeners that accounts have changed.
	this.broadcast();
	this._notifyDefaultAccountsListeners(this.defaultAccountsList, oldDefaultAccountsList);
	
	if (this._onReady) {
		this._gotAccounts = true;
		this._onReady();
		this._onReady = undefined;
	}
};

AccountList.defaultAccountPriority = ["com.palm.eas", "com.palm.google"];

AccountList.prototype.isReady = function () {
	return this._gotAccounts;
};

AccountList.prototype.isFetchingAccounts = function () {
	return this._fetchingAccounts;
};


//@ sourceURL=contacts.ui/Utilities/EventListenerManager.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports*/

var EventListenerManager = exports.EventListenerManager = function () {
	var _listeners = [];

	return {
		addListener: function (element, event, callback, useCapture) {
			_listeners.push({
				element: element,
				event: event,
				callback: callback,
				capture: useCapture
			});
			element.addEventListener(event, callback, useCapture);
		},
		
		destroyListeners: function () {
			var listener,
				i,
				len = _listeners.length;
				
			if (!len) {
				return;
			}

			for (i = 0; i < len; i += 1) {
				listener = _listeners[i];
				listener.element.removeEventListener(listener.event, listener.callback, listener.capture);
			}
			_listeners = [];
		}
	};
};

//@ sourceURL=contacts.ui/Utilities/PersonAsyncDecorator.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Mojo, Contacts, _, PersonImStatusAsyncDecorator, PersonPhotoAsyncDecorator */

// Class for decorating a *group* of list items with im status and photos.
// The list widget asks us to allocate one of these for each page of items, and gives us a way to update 
// the data for the items in the page when the im status changes.  When the page is no longer 
// needed, we're cancelled, and can release any resources we were holding on to.
var PersonAsyncDecorator = exports.PersonAsyncDecorator = function (items, decorateItemsCallback) {
	Mojo.Log.info("Created PersonAsyncDecorator for %d items.", items.length);
	
	//sanity-check the params
	if (!items || !_.isArray(items) || items.length === 0 || !decorateItemsCallback || !_.isFunction(decorateItemsCallback)) {
		Mojo.Log.error("Tried to create PersonAsyncDecorator with invalid params. %s %s", Contacts.Utils.stringify(items), Contacts.Utils.stringify(decorateItemsCallback));
		return;
	}
	
	this._decorateItemsCallback = decorateItemsCallback;
	this._decoratorCache = {};
	
	//set up both of the individual decorators.  just give them the callback we received from the 
	//list - there's no need to wrap it.
	this.imStatusDecorator = new PersonImStatusAsyncDecorator(items, this.callback.bind(this));
	this.photoDecorator = new PersonPhotoAsyncDecorator(items, this.callback.bind(this));
};

// Called by list widget when this group of items is being thrown away.
// Cancels the watch so that it will no longer receive updates to the IM status for its person
PersonAsyncDecorator.prototype.cancel = function () {
	Mojo.Log.info("Cancelling PersonAsyncDecorator for %d items.", this._personIds && this._personIds.length);
	
	if (this.imStatusDecorator) {
		this.imStatusDecorator.cancel();
		this.imStatusDecorator = undefined;
	}
	if (this.photoDecorator) {
		this.photoDecorator.cancel();
		this.photoDecorator = undefined;
	}
	
	this._decoratorCache = {};
};

PersonAsyncDecorator.prototype.callback = function (newDecorators) {
	var that = this,
		extendedDecorators;
	
	if (!newDecorators || !Array.isArray(newDecorators)) {
		Mojo.Log.error("PersonAsyncDecorator.decorateItemsCallback called with non-array arg. %s", Contacts.Utils.stringify(newDecorators));
		return;
	}
	
	extendedDecorators = newDecorators.map(function (newDecorator) {
		var personId = newDecorator._id,
			completeDecorator;
		
		//get the old one (or create a new one) and update it with the new fields
		completeDecorator = that._decoratorCache[personId] || {};
		_.extend(completeDecorator, newDecorator);
		
		//store the updated decorator in the cache and return it so that we send it to the list
		that._decoratorCache[personId] = completeDecorator;
		return completeDecorator;
	});
	
	//now call the list's callback with the extended decorators
	this._decorateItemsCallback(extendedDecorators);
};


//@ sourceURL=contacts.ui/Utilities/PersonImStatusAsyncDecorator.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Mojo, Contacts, _ */

// Class for monitoring im status for a *group* of list items.
// The list widget asks us to allocate one of these for each page of items, and gives us a way to update 
// the data for the items in the page when the im status changes.  When the page is no longer 
// needed, we're cancelled, and can release any resources we were holding on to.
var PersonImStatusAsyncDecorator = exports.PersonImStatusAsyncDecorator = function (items, decorateItemsCallback) {
	Mojo.Log.info("Created PersonImStatusAsyncDecorator for %d items.", items.length);
	
	//sanity-check the params
	if (!items || !_.isArray(items) || items.length === 0 || !decorateItemsCallback || !_.isFunction(decorateItemsCallback)) {
		Mojo.Log.error("Tried to create PersonImStatusAsyncDecorator with invalid params. %s %s", Contacts.Utils.stringify(items), Contacts.Utils.stringify(decorateItemsCallback));
		return;
	}
	
	var that = this,
		query;
	
	//set up the objects we need for the items
	this._personIds = [];
	this._currentStatusMap = {};
	items.forEach(function (item) {
		that._personIds.push(item._id);
		that._currentStatusMap[item._id] = Contacts.IMAddress.STATUS.NO_PRESENCE;
	});
	
	query = {
		select: ["personId", "personAvailability"],
		from: "com.palm.imbuddystatus:1",
		where: [{
			prop: "personId",
			op: "=",
			val: this._personIds
		}]
	};
	
	this._decorateItemsCallback = decorateItemsCallback;
	
	this._dbWatcher = new Contacts.Utils.DBWatcher(query, this._watchFireHandler.bind(this), true);
};

// Called by list widget when this group of items is being thrown away.
// Cancels the watch so that it will no longer receive updates to the IM status for its person
PersonImStatusAsyncDecorator.prototype.cancel = function () {
	Mojo.Log.info("Cancelling PersonImStatusAsyncDecorator for %d items.", this._personIds && this._personIds.length);
	
	if (this._dbWatcher) {
		this._dbWatcher.cancel();
		this._dbWatcher = undefined;
	}
	
	this._personIds = undefined;
	this._currentStatusMap = undefined;
};

PersonImStatusAsyncDecorator.prototype._watchFireHandler = function (queryResults) {
	var that = this,
		personsSeen = {},
		decorators = [];
	
	//TODO: this doesn't work when the buddystatus object is deleted
	//need to incDel on the personId query and index
	
	queryResults = queryResults || [];
	queryResults.forEach(function (buddyStatus) {
		var personId = buddyStatus.personId,
			oldStatus,
			newStatus;
		
		//if we've already gotten an entry for this person, skip them.  that way we don't end up 
		//with multiple entries in the decorators array for a single person
		if (personsSeen[personId]) {
			return;
		}
		personsSeen[personId] = true;
		
		oldStatus = that._currentStatusMap[personId];
		newStatus = buddyStatus.personAvailability;
		
		// only push to decorators if it actually changed.  This saves a lot of work if there 
		//is no change in status or they're not logged in, since we assume NO_PRESENCE on init
		if (newStatus !== undefined && newStatus !== oldStatus) {
			Mojo.Log.info("PersonImStatusAsyncDecorator: person %s now has presence %d", personId, newStatus);
			
			that._currentStatusMap[personId] = newStatus;
			decorators.push({
				_id: personId,
				imStatus: newStatus,
				imStatusClassName: Contacts.PersonDisplay.getImStatusClassName(newStatus)
			});
		}
	});
	
	//if we have any actual changes, call the callback
	if (decorators.length > 0) {
		this._decorateItemsCallback(decorators);
	}
};


//@ sourceURL=contacts.ui/Utilities/PersonPhotoAsyncDecorator.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Mojo, Contacts, _, Foundations */

// Class for loading photos for a *group* of list items.
// The list widget asks us to allocate one of these for each page of items, and gives us a way to update 
// the data for the items in the page when the photo changes.  When the page is no longer 
// needed, we're cancelled, and can release any resources we were holding on to.
var PersonPhotoAsyncDecorator = exports.PersonPhotoAsyncDecorator = function (items, decorateItemsCallback) {
	Mojo.Log.info("Created PersonPhotoAsyncDecorator for %d items.", items.length);
	
	//sanity-check the params
	if (!items || !_.isArray(items) || items.length === 0 || !decorateItemsCallback || !_.isFunction(decorateItemsCallback)) {
		Mojo.Log.error("Tried to create PersonPhotoAsyncDecorator with invalid params. %s %s", Contacts.Utils.stringify(items), Contacts.Utils.stringify(decorateItemsCallback));
		return;
	}
	
	//console.log("\n\n\n---------------->>>>>>>>>>>>>>> decorate items count: " + Contacts.Utils.stringify(items.length) + "\n\n\n");
	this._decorateItems(items, decorateItemsCallback);
};

// Called by list widget when this group of items is being thrown away.
// Cancels the future so that the remainder of the processing is thrown away and the callback is never called
PersonPhotoAsyncDecorator.prototype.cancel = function () {
	Mojo.Log.info("Cancelling PersonPhotoAsyncDecorator for %d items.", this._personIds && this._personIds.length);
	
	if (this._future) {
		this._future.cancel();
		this._future = undefined;
	}
};


PersonPhotoAsyncDecorator.prototype._decorateItems = function (items, decorateItemsCallback) {
	var that = this;
	
	this._future = Foundations.Control.mapReduce({
		map: function (person) {
			//console.log("\n\n\n---------------->>>>>>>>>>>>>>> decorating a person \n\n\n");
			
			//origListPhotoPath is the version attached by PersonDisplayLite in the (regular/non-async) decorator
			var origListPhotoPath = person && person.listPhotoPath,
				mapFuture = Contacts.PersonPhotos.getPhotoPath(person.photos, Contacts.PersonPhotos.TYPE.LIST, true);
			
			mapFuture.then(that, function () {
				try {
					var filepath = mapFuture.result;
					
					//console.log("\n\n\n---------------->>>>>>>>>>>>>>> filepath: " + Contacts.Utils.stringify(filepath) + "\n\n\n");
					
					mapFuture.result = {
						_id: person._id,
						listPhotoPath: filepath,
						origListPhotoPath: origListPhotoPath
					};
				} catch (ex) {
					Mojo.Log.error("PersonPhotoAsyncDecorator: exception! " + Contacts.Utils.stringify(ex));
					
					//return an empty object here so that this item will be filtered out of the decorators array down below
					mapFuture.result = {};
				}
			});
			
			return mapFuture;
		}
	}, items);
	
	this._future.then(this, function () {
		//TODO: this map-filter combo could be replaced by a reduce
		var wrappedDecorators = this._future.result,
			decorators = wrappedDecorators.map(function (wrappedDecorator) {
				return wrappedDecorator.result;
			});
		
		//we want to filter out those that don't have a photo path and those for which the path is 
		//the same as the original one (that is, those that weren't refetched)
		decorators = decorators.filter(function (decorator) {
			return decorator && decorator.listPhotoPath && decorator.listPhotoPath !== decorator.origListPhotoPath;
		});
		
		//console.log("\n\n\n---------------->>>>>>>>>>>>>>> decorators count: " + Contacts.Utils.stringify(decorators.length) + "\n\n\n");
		//console.log("\n\n\n---------------->>>>>>>>>>>>>>> decorators[0]: " + Contacts.Utils.stringify(decorators[0]) + "\n\n\n");
		
		decorateItemsCallback(decorators);
	});
};


//@ sourceURL=contacts.ui/Utilities/ImAddressStatusWatcher.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Mojo, Contacts, _ */

// Class for monitoring im status for a *group* of list items.
// The list widget asks us to allocate one of these for each page of items, and gives us a way to update 
// the data for the items in the page when the im status changes.  When the page is no longer 
// needed, we're cancelled, and can release any resources we were holding on to.
var ImAddressStatusWatcher = function (imAddress, type, itemWatchCallback) {
	Mojo.Log.info("Created ImAddressStatusWatcher for item %s on %s.", imAddress, type);
	
	//sanity-check the params
	if (!imAddress || !_.isString(imAddress) || !type || !_.isString(type) || !itemWatchCallback || !_.isFunction(itemWatchCallback)) {
		Mojo.Log.error("Tried to create ImAddressStatusWatcher with invalid params. %s %s %s", imAddress, type, itemWatchCallback);
		return;
	}
	
	//set up the objects we need for the query
	this._imAddress = imAddress;
	this._type = type;
	this._currentStatus = Contacts.IMAddress.STATUS.NO_PRESENCE;
	
	var query = {
		select: ["username", "serviceName", "availability"],
		from: "com.palm.imbuddystatus:1",
		where: [{
			prop: "username",
			op: "=",
			val: this._imAddress
		}, {
			prop: "serviceName",
			op: "=",
			val: this._type
		}]
	};
	
	this._itemWatchCallback = itemWatchCallback;
	
	this._dbWatcher = new Contacts.Utils.DBWatcher(query, this.watchFireHandler.bind(this), true);
};

// Called by list widget when this group of items is being thrown away.
// Cancels the watch so that it will no longer receive updates to the IM status for its person
ImAddressStatusWatcher.prototype.cancel = function () {
	Mojo.Log.info("Cancelling ImAddressStatusWatcher for item %s on %s.", this._imAddress, this._type);
	
	if (this._dbWatcher) {
		this._dbWatcher.cancel();
		this._dbWatcher = undefined;
	}
	
	this._imAddress = undefined;
	this._type = undefined;
	this._currentStatus = undefined;
};

ImAddressStatusWatcher.prototype.watchFireHandler = function (queryResults) {
	if (!queryResults || !_.isArray(queryResults)) {
		Mojo.Log.info("Invalid ImAddressStatusWatcher queryResults %s.", Contacts.Utils.stringify(queryResults));
		return;
	}
	
	//find the "best" status out of all the results (the one that is "most online", aka the smallest int)
	var newStatus = queryResults.reduce(function (curBestStatus, curQueryResult) {
		if (curQueryResult.availability < curBestStatus) {
			return curQueryResult.availability;
		} else {
			return curBestStatus;
		}
	}, Contacts.IMAddress.STATUS.NO_PRESENCE);
	
	// only call the callback if it actually changed.  This saves a lot of work if there 
	//is no change in status or they're not logged in, since we assume NO_PRESENCE on init
	if (newStatus !== undefined && newStatus !== this._currentStatus) {
		Mojo.Log.info("ImAddressStatusWatcher: item %s on %s now has presence %d", this._imAddress, this._type, newStatus);
		
		this._currentStatus = newStatus;
		this._itemWatchCallback({
			imAddress: this._imAddress,
			serviceName: this._type,
			type: this._type,
			imStatus: newStatus,
			imStatusClassName: Contacts.PersonDisplay.getImStatusClassName(newStatus)
		});
	}

};

ImAddressStatusWatcher.prototype.getCurrentStatus = function () {
	return {
		imAddress: this._imAddress,
		type: this._type,
		imStatus: this._currentStatus,
		imStatusClassName: Contacts.PersonDisplay.getImStatusClassName(this._currentStatus)
	};
};


//@ sourceURL=contacts.ui/Utilities/MultipleListManager.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Assert, _ */

/**
 * This is designed to be used with TinyList in the details view. No other considerations have
 * been made for this class.
 * 
 * Possible additions: removeListConfig(), clearListItems(), getListItems()
 * @Class
 */

var MultipleListManager = function () {
	// This object contains a property for each list where the listName is the hash key
	// Each list has the following properties:
	//	configs[<listName>] = {
	//		"attributes": {object},
	//		"model": {object},
	//		"div": {HTMLElement#Div},
	//		"update": {function},
	//		"tapEventListenerDestructor": {function}
	//	};
	var configs = {};
	
	return {
		addListConfig: function (listName, attributes, model, div, updateFn, tapEventListenerDestructor) {
			Assert.requireString(listName, "addListConfig missing listName");
			Assert.requireDefined(attributes, "addListConfig missing attributes");
			Assert.requireDefined(model, "addListConfig missing model");
			Assert.requireDefined(updateFn, "addListConfig missing update function");
			// make sure there is not already an entry with the same listName
			var listConfig = configs[listName];
			Assert.requireFalse(listConfig, "addListConfig a list with the same listName already exists");
			configs[listName] = {
				"list": attributes,
				"model": model,
				"div": div,
				"update": updateFn,
				"tapEventListenerDestructor": tapEventListenerDestructor
			};
		},
		
		updateListItems: function (listName, items) {
			Assert.requireString(listName, "updateListItems requires a listName");
			Assert.requireArray(items, "updateListItems requires an items array");
			
			var listConfig = configs[listName];
			Assert.require(listConfig, "unable to find listConfig for listName: " + listName);
			
			listConfig.model.items = items;
			listConfig.update(listConfig.model.items);
		},
		
		/**
		 * Handles cleanup (kills event listeners)
		 */
		destroy: function () {
			Object.keys(configs).forEach(function (listName) {
				var item = configs[listName];
				if (item.tapEventListenerDestructor && _.isFunction(item.tapEventListenerDestructor)) {
					item.tapEventListenerDestructor();
				}
			});
		}
	};
};

//@ sourceURL=contacts.ui/Utilities/MenuHandler.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, $LL, RB */

var MenuHandler = {
	createAppMenu: function (that, menuModel) {
		that.goToPrefs = this.goToPrefs;
		that.handleCommand = this.handleCommand;
		var items = [Mojo.Menu.editItem];
		if (menuModel) {
			items = items.concat(menuModel);
		}
		items.push(MenuHandler.prefsItem);
		items.push({
			label: $LL('Help'),
			command: Mojo.Menu.helpCmd
		});
		that.appMenuModel = that.appMenuModel || {
			visible: true,
			label: RB.$L('Contacts Menu'),
			items: items
		};
		that.appMenuModel.items = items;
		that.controller.setupWidget(Mojo.Menu.appMenu, {
			omitDefaultItems: true
		}, that.appMenuModel);
	},
	
	goToPrefs: function () {
		// TODO: this scene should be in the contacts.ui LF, but currently it is still in the app
		this.controller.stageController.pushScene('prefs');
	},
	
	handleCommand: function (event) {
		// handle menu button command events
		if (event.type === Mojo.Event.command) {
			if (event.command === Mojo.Menu.prefsCmd) {
				Mojo.Event.stop(event);
				this.goToPrefs();
				return;
			}
			if (event.command === MenuHandler.deleteContact) {
				Mojo.Event.stop(event);
				this.handleDelete();
				return;
			}
			if (event.command === MenuHandler.addToLauncher) {
				Mojo.Event.stop(event);
				this.handleAddToLauncher();
				return;
			}
			if (event.command === MenuHandler.noop) {
				Mojo.Event.stop(event);
				return;
			}
			
			if (event.command === MenuHandler.linkProfiles) {
				Mojo.Event.stop(event);
				this.handleClipMore();
				return;
			}
			if ((typeof event.command) === "function") {
				event.command();
				Mojo.Event.stop(event);
			}
			
			if (event.command === MenuHandler.editContact) {
				this.handleEdit();
				Mojo.Event.stop(event);
				return;
			}
			
			if (event.command === MenuHandler.setSpeedDial) {
				this.setSpeedDial();
				Mojo.Event.stop(event);
				return;
			}
			
			if (event.command === MenuHandler.addReminder) {
				this.handleReminder();
				Mojo.Event.stop(event);
				return;
			}
			
			if (event.command === MenuHandler.addToSIM) {
				this.addToSIM();
				Mojo.Event.stop(event);
				return;
			}
			
			if (event.command === MenuHandler.copyContact) {
				this.copyDetailsToClipboard();
				Mojo.Event.stop(event);
				return;
			}
		}
	}
};

MenuHandler.editContact = "edit-contact";
MenuHandler.deleteContact = "deleteContact";
MenuHandler.preferences = "preferences";
MenuHandler.noop = "noop";
MenuHandler.addToLauncher = "addToLauncher";
MenuHandler.linkProfiles = "linkProfiles";
MenuHandler.setSpeedDial = "setSpeedDial";
MenuHandler.addReminder = "addReminder";
MenuHandler.addToSIM = "addToSIM";
MenuHandler.copyContact = "copyContact";
MenuHandler.prefsItem = {
	label: RB.$L("Preferences & Accounts"),
	command: Mojo.Menu.prefsCmd,
	disabled: true
};

MenuHandler.launchHelp = function () {
	var request = new Mojo.Service.Request("palm://com.palm.applicationManager", {
		method: "open",
		parameters: {
			id: 'com.palm.app.help',
			params: {
				target: "http://help.palm.com/contacts/index.html"
			}
		}
	});
};


//@ sourceURL=contacts.ui/Utilities/OpenWith.js

/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global RB, Mojo, _ */

/*
 * Submenu for displaying the list of applications
 * that are capable of opening the type of data
 * specified.
 * 
 * Required parameters to constructor are
 * - isMimeType -> <bool> does this type of data
 *					require mime type - true
 *					does this type of data require
 *					a url type - false 
 * - urlOrMime	-> the url or mime type that this
 *					data could correspond to.
 * - controller -> the scene controller that contains
 *					the mojo element that this menu corresponds
 *					to.
 * - listId		-> the id of the list object the submenu
 *					should be attached to.
 * 
 * Assuming the mojo element exists at the time of creating
 * this menu object, all you need to do is create the object.
 * Otherwise you will need to add a listener to the object
 * that should get the open with menu. The method the event
 * should be binded to is shouldOpenOpenWith.
 */
var OpenWithMenu = function (params) {
	this.isMimeType = params.isMimeType;
	this.urlOrMime = params.urlOrMime;
	this.controller = params.controller;
	this.listId = params.listId;
	
	this.pushLoading = false;
	this.loadingModel = [
		{
			label: RB.$L("Copy"),
			command: "openwith-copyCommand"
		}, {
			label: RB.$L("Open With")
		}, {
			label: RB.$L("Loading..."),
			command: "loading"
		}
	];
	this.handlers = [];
	this.bareData = "";
	this.data = null;
	this.originalEventForShowMenu = null;
	this.mimeSystemService = "palm://com.palm.applicationManager";
	this.mimeAppListMethod = "listAllHandlersForMime";
	this.urlAppListMethod = "listAllHandlersForUrl";
	this.launchDefaultFailed = RB.$L("Launching Default Application Failed");
	this.openWithTitle = RB.$L("Open With");
	
	this.shouldOpenOpenWithBound = this.shouldOpenOpenWith.bind(this);
	this.controller.listen(this.listId, Mojo.Event.listTap, this.shouldOpenOpenWithBound);
};

// Determine if the orange key is being pressed
//
// If it is we want to show the open with menu
// If it is not we want to lauch the default app
OpenWithMenu.prototype.shouldOpenOpenWith = function (e) {
	var fullAddress = e.item.getDisplayValue(),
		data = {
			target: this.urlOrMime + fullAddress.replace(/[\n\r]/g, " ")
		};
	
	this.bareData = fullAddress;
	
	// Was orange pressed
	if (e.originalEvent.down.altKey) {
		this.showMenu(e.originalEvent, data);
	} else {
		// Orange was not pressed so launch default
		this.launchDefault(data);
	}
};

OpenWithMenu.prototype.destroy = function () {
	this.controller.stopListening(this.listId, Mojo.Event.listTap, this.shouldOpenOpenWithBound);
};

// Show the open with menu
//
// Fetch the list of apps that can
// handle the urlOrMime type specified
OpenWithMenu.prototype.showMenu = function (event, data) {
	var params, method, request;
	
	// Capture the original event that caused
	// this menu to be displayed. This
	// way we can have the open with menu
	// appear near the original event after
	// the list of apps loads
	this.originalEventForShowMenu = event;
	
	// Display the loading menu
	// until the list of open with 
	// apps is returned
	this.showLoadingMenu(event);
	
	// If any data was sent to this method
	// save it
	if (data) {
		this.data = data;
	}
	
	params = {};
	method = "";
	
	if (this.isMimeType) {
		params = {
			mime: this.urlOrMime
		};
		
		method = this.mimeAppListMethod;
	} else {
		params = {
			url: this.urlOrMime
		};
		
		method = this.urlAppListMethod;
	}
	
	// Get the list of apps that are capable
	// of handling this type of data
	request = new Mojo.Service.Request(this.mimeSystemService, {
		method: method,
		parameters: params,
		onSuccess: function (e) {
			// Parse out the results and show
			// the list of apps in the submenu.
			// 
			// If there was an error it will display
			// an alert saying the mime system
			// failed.
			this.listAllHandlersSuccess(e, event);
		}.bind(this),
		onFailure: function (e) {
			// Mime system is not reachable.
			// Display alert indicating this.
			this.displayMimeSystemFailed();
		}.bind(this)
	});
};

OpenWithMenu.prototype.showLoadingMenu = function (event) {
	if (this.pushLoading) {
		// Hide any existing open with menus
		// if they exist
		if (this.popUpMenu) {
			this.popUpMenu.mojo.close();
		}
		
		// Pop up the menu that will hold the list of
		// apps to open with. First display the loading
		// menu. The list of apps will display
		// as soon as they are finished loading.
		this.popUpMenu = this.controller.popupSubmenu({
			onChoose: this.clickedMenu.bind(this),
			placeNear: event.target,
			items: this.loadingModel
		});
	}
};

// Parse and display the apps that are
// able to open this data.
OpenWithMenu.prototype.listAllHandlersSuccess = function (result, originalEvent) {
	if (this.popUpMenu) {
		this.popUpMenu.mojo.close();
	}
	
	
	if (result.returnValue) {
		// Reset the handlers list
		// since we have new data
		this.pushHandlersInitialItems();
		
		// There are two possible handler
		// types that can be returned.
		// Check for either and assign
		// result as the valid handler.
		//
		// We don't need any of the data
		// above the handlers anymore
		// so we just write over result
		if (result.redirectHandlers) {
			result = result.redirectHandlers;
		} else if (result.resourceHandlers) {
			result = result.resourceHandlers;
		}
		
		// If for some reason an app does not
		// have an app name associated with 
		// it we will just use the appId.
		if (result.activeHandler.appName) {
			result.activeHandler.label = result.activeHandler.appName;
		} else {
			result.activeHandler.label = result.activeHandler.appId;
		}
		
		// Set the command of the handler to be equal to the appId.
		// This allows us to use the value that was clicked on the
		// open with menu.
		result.activeHandler.command = result.activeHandler.appId;
		this.handlers.push(result.activeHandler);
		
		// If there is more than one app that can handle this
		// data, those handlers will be set in alternates.
		if (result.alternates) {
			// Go through each alternate, set it up for
			// display in the open with menu, and add it
			// to the handlers array.
			result.alternates.forEach(function (handler) {
				if (handler.appName) {
					handler.label = handler.appName;
				} else {
					handler.label = handler.appId;
				}
				handler.command = handler.appId;
				this.handlers.push(handler);
			}.bind(this));
		}
		
		// Display the list of apps that can open 
		// this type of data.
		this.popUpMenu = this.controller.popupSubmenu({
			onChoose: this.clickedMenu.bind(this),
			placeNear: originalEvent.target,
			items: this.handlers
		});
	} else {
		// The call to the mime system
		// failed because of an error in
		// the system. Display an alert
		// saying this.
		this.displayMimeSystemFailed();
	}
};

// Sets the initial items that need to
// be in the Open With submenu
OpenWithMenu.prototype.pushHandlersInitialItems = function () {
	this.handlers = [];
	this.handlers.push({ label: RB.$L("Copy"), command: "openwith-copyCommand" });
	this.handlers.push({ label: RB.$L("Open With") });
};

// Get the app id of the default
// application for this mime type or url
OpenWithMenu.prototype.getActiveHandlerForDefault = function (result) {
	if (result.returnValue) {
		// Return the appId of the default handler 
		// from the Mime systems response
		if (result.redirectHandlers) {
			return result.redirectHandlers.activeHandler.appId;
		} else if (result.resourceHandlers) {
			return result.resourceHandlers.activeHandler.appId;
		}
					
	} else {
		// The call to the mime system
		// failed because of an error in
		// the system. Display an alert
		// saying this.
		this.displayFailedMessage(this.launchDefaultFailed);
		return false;
	}
};

// Handle a click on the open with menu
// 
// If it was not a loading click,
// launch the app chosen
OpenWithMenu.prototype.clickedMenu = function (appId) {		
	if (appId) {
		if (appId === "openwith-copyCommand") {
			this.controller.stageController.setClipboard(this.bareData);
		}
		else if (appId !== "loading") {
			this.launchApp(appId, this.data);
		} 
		else {
			this.showLoadingMenu(this.originalEventForShowMenu);
		}
	} 
};

// Open the default application with
// the data specified
OpenWithMenu.prototype.launchDefault = function (data) {
	var params, method, request;
	
	this.data = data;
	
	params = {};
	method = "";
	
	if (this.isMimeType) {
		params = {
			mime: this.urlOrMime
		};
		
		method = this.mimeAppListMethod;
		
	} else {
		params = {
			url: this.urlOrMime
		};
		
		method = this.urlAppListMethod;
	}
	
	// Get the default application from the
	// mime system.
	request = new Mojo.Service.Request(this.mimeSystemService, {
		method: method,
		parameters: params,
		onSuccess: function (e) {
			var defaultHandler = this.getActiveHandlerForDefault(e);
			if (defaultHandler) {
				this.launchApp(defaultHandler, this.data);
			} else {
				this.displayFailedMessage(this.launchDefaultFailed);
			}
		}.bind(this),
		onFailure: function (e) {
			this.displayFailedMessage(this.launchDefaultFailed);
		}.bind(this)
	});
};

// Open the app with the appid
// and data that was specified
OpenWithMenu.prototype.launchApp = function (appid, data) {
	var tempData, request;
	
	// HACK for getting around mapto should be maploc in google maps
	if (appid === "com.palm.app.maps") {
		tempData = data.target.substr(this.urlOrMime.length);
		data.target = "maploc:" + tempData;
	}
	///////////////////////////////////////////////////////////////
	
	request = new Mojo.Service.Request(this.mimeSystemService, {
		method: 'open',
		parameters: {
			id: appid,
			params: data
		},
		onFailure: function (e) {
			this.displayAppLaunchFailed(appid);
		}.bind(this)
	});
};

// Display in an alert dialog that
// launching the specifed app failed
OpenWithMenu.prototype.displayAppLaunchFailed = function (appid) {
	var template, appDisplay, message;
	
	template = new Mojo.Format.Template(RB.$L("#{appName} Failed To Launch"));
	appDisplay = this.getAppNameFromHandlers(appid);
	message = template.evaluate({ appName: appDisplay });
	
	this.displayFailedMessage(message);
};

// Display in an alert dialog that
// the Mime system failed to 
// return without an error
OpenWithMenu.prototype.displayMimeSystemFailed = function () {
	this.displayFailedMessage(RB.$L("Failed To Get Applications List"));
};

// Display in an alert dialog that
// the Mime system failed because
// of the reason given in message
OpenWithMenu.prototype.displayFailedMessage = function (message) {
	if (this.popUpMenu) {
		this.popUpMenu.mojo.close();
	}

	this.controller.showAlertDialog({
		message: message,
		title: this.openWithTitle,
		choices: [{ label: RB.$L("OK") }]
	});
};

// Pull out the app name from the 
// handlers array.
//
// If the appid is not found or the
// handler array is null, return
// the appid as it was given to 
// this method.
OpenWithMenu.prototype.getAppNameFromHandlers = function (appid) {
	if (!this.handlers) {
		return appid;
	}
	
	// Run through the handlers trying
	// to find the handler that corresponds
	// to the appid passed to the function.
	var toReturn = _.detect(this.handlers, function (handler) { 
		if (handler.appId === appid) {
			return handler;
		}
	});
	
	// If there was no corresponding
	// handler, just return the appid
	// given to the function.
	if (!toReturn) {
		return appid;
	}
	
	// There is a possibilty that the mime
	// system will not return the label for
	// a handler. If this is the case
	// we must at least return the handler's appId
	if (toReturn.label) {
		return toReturn.label;
	} else {
		return toReturn.appId;
	}
};


//@ sourceURL=contacts.ui/Utilities/tiny-list.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Mojo, LIB_ROOT*/

/*
 * TinyList: the non-widget widget for simple tappable lists
 */

var TinyList = {};

/**
 * 
 * @param {Object} div
 * @param {Object} model
 * @param {Object} handler
 * @returns {function} function to destroy the event listener
 */
TinyList.setupTapHandler = function (div, model, handler) {
	var wrappedHandler = TinyList._handleListTap.bind(undefined, model, div, handler);
	Mojo.Event.listen(div, Mojo.Event.tap, wrappedHandler);
	return Mojo.Event.stopListening.bind(this, div, Mojo.Event.tap, wrappedHandler);
};

TinyList.renderData = function (div, attrs, items) {
	if (!items || items.length === 0) {
		div.innerHTML = "";
		return;
	}
	
	TinyList._processListItems(items);
	
	var list = Mojo.View.render({
		templateRoot: attrs.templateRoot,
		template: attrs.itemTemplate,
		collection: items
	});
	if (attrs.listTemplate) {
		list = Mojo.View.render({
			templateRoot: attrs.templateRoot,
			template: attrs.listTemplate,
			object: {
				listElements: list
			}
		});
	}
	
	div.innerHTML = list;
};

TinyList._processListItems = function (items) {
	if (!items || items.length === 0) {
		return;
	}
	
	if (items.length === 1) {
		// apply single
		items[0].rowClass = 'single';
	} else {
		// apply first and last
		items[0].rowClass = 'first';
		items[items.length - 1].rowClass = 'last';
	}
	
	for (var i = 0; i < items.length; i += 1) {
		items[i].listIndex = i;
	}
	
};

TinyList._handleListTap = function (model, listDiv, handler, e) {
	e.originalEvent = e;
	
	var indexAndElem = TinyList._getIndexOfTap(e.target, listDiv);
	if (!indexAndElem.index) {
		Mojo.Log.error("Couldn't find x-list-index target for list tap");
		return;
	}
	e.item = model.items[indexAndElem.index];
	//tack on this itemElement thing that tells me the top-level element for this item
	e.itemElement = indexAndElem.element;
	
	handler(e);
};

TinyList._getIndexOfTap = function (element, toplevel) {
	var index = element.getAttribute('x-list-index');
	
	while (index === null && element && element !== toplevel) {
		element = element.parentElement;
		index = element.getAttribute('x-list-index');
	}
	return {
		index: index,
		element: element
	};
};


//@ sourceURL=contacts.ui/Utilities/Utils.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global _, Mojo, LIB_ROOT, Contacts */

var Utils = {
	loadSceneStyles: function (sceneController, stylesheetPath, identifier) {
		// the stageController version of this function is a bit
		// out of control. there is no equivalent unload however
		// so we use the stagecontroller version in that case.
		if (!_.isArray(stylesheetPath)) {
			stylesheetPath = [stylesheetPath];
		}
		for (var i = 0; i < stylesheetPath.length; i += 1) {
			Mojo.loadStylesheet(sceneController.document, stylesheetPath[i], LIB_ROOT, identifier);
			Mojo.Locale._loadLocalizedStylesheet(sceneController.document, stylesheetPath[i], LIB_ROOT, identifier);
			//Utils.loadSceneStyle(sceneController.document, stylesheetPath[i], identifier);
		}
	},
	
	unloadSceneStyles: function (sceneController, stylesheetPath, identifier) {
		if (!_.isArray(stylesheetPath)) {
			stylesheetPath = [stylesheetPath];
		}
		//console.log("head: " + sceneController.document.getElementsByTagName("head")[0].innerHTML);
		for (var i = 0; i < stylesheetPath.length; i += 1) {
			// The framework unloadStylesheet function removes all instances of a stylesheet
			// This is bad when the same stylesheet has been included as a result of pushing the same scene
			// multiple times.  In contacts we often push list view, details view, list view (people picker)
			
			//stageController.unloadStylesheet(stylesheetPath[i]);
			Utils.unloadAllInstancesOfSceneStyle(sceneController, stylesheetPath[i], identifier);
		}
	},
	
	// borrowed this code from the framework and modified it to take an identifier param
	unloadAllInstancesOfSceneStyle: function (sceneController, path, identifier) {
		var i,
			theDocument = sceneController.document,
			links = theDocument.querySelectorAll('link[type="text/css"][href$="' + path + '"]'),
			head = theDocument.querySelector('head'),
			link,
			linkAttr;
			
		if (!head) {
			Mojo.Log.warn("No <head> element!");
			return;
		}
		
		for (i = 0; i < links.length; i += 1) {
			link = links[i];
			if (identifier) {
				linkAttr = link.getAttribute("x-mojo-css-identifier");
				if (linkAttr && linkAttr === identifier) {
					link.disabled = true;
					head.removeChild(link);
				}
			} else {
				link.disabled = true;
				head.removeChild(link);
			}
		}
	},
	
	// Poor man's formatting support for new list.
	// Patches the given datasource assistant to call the given formatter function with each object used in the list.
	formatDataSourceAssistant: function (dsa, formatter) {
		var getItemsToRenderOriginal;
		
		getItemsToRenderOriginal = dsa.getItemsToRender.bind(dsa);
		
		dsa.getItemsToRender = function (response, callback) {
			getItemsToRenderOriginal(response, function (items) {
				if (items) {
					for (var i = 0; i < items.length; i += 1) {
						items[i] = formatter(items[i]);
					}
				}
				callback(items);
			});
		};
		
		return dsa;
	},
	
	getTemplatePath: function (template) {
		return "templates/" + template;
	},
	
	getStylesheetPath: function (stylesheet) {
		return "stylesheets/" + stylesheet;
	},
	
	/**
	 * 
	 * @param {Object} item					A single function or an array of functions
	 * @param {Object} args
	 * @param {Object} scope
	 * @param {Object} functionIsFirstArg
	 */
	executeFunctionIfExists: function (item, argsArray, scope) {
		var i,
			fn;
			
		if (_.isArray(item)) {
			
		} else {
			item = [item];
		}
		
		for (i = 0; i < item.length; i += 1) {
			fn = item[i];
			if (fn && _.isFunction(fn)) {
				fn.apply(scope, argsArray);	
			}
		}
	},
	
	getCPsForPersonForType: function (person, type, filter) {
		var mappingFunction = function (array, filter) {
			if (filter) {
				var toReturn = [];
				
				array.forEach(function (contactPoint) {
					if (contactPoint.getType() === filter) {
						toReturn.push(contactPoint);
					}
				});
				
				return toReturn;
			} else {
				return array;
			}
		};

		switch (type) {
		case Contacts.ContactPointTypes.PhoneNumber:
			return mappingFunction(person.getPhoneNumbers().getArray(), filter);
		case Contacts.ContactPointTypes.EmailAddress:
			return mappingFunction(person.getEmails().getArray(), filter);
		case Contacts.ContactPointTypes.IMAddress:
			return mappingFunction(person.getIms().getArray(), filter);
		default:
			return [];
		}
	}
};


//@ sourceURL=contacts.ui/Utilities/gal-helper.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global Contact, Email, PhoneNumber, Address, window, _, Mojo */

// TODO: clean up the contract between this and the list assistant
// should probably be widget-like-- the scene includes some generic GAL div and here it's fleshed out
// and behaviors are hooked up to it.  The main interaction with the rest of the scene is just
// reading the filter string, updating the count, and returning a GAL contact to view details. 

var GALHelper = function (assistant) {
	this.asst = assistant;
	
	this.toggler = this.asst.controller.get("GalDrawerOpener");
	this.counter = this.asst.controller.get("GALCountResults");
	this.counterContainer = this.asst.controller.get("GALCountContainer");
	this.listDiv = this.asst.lookups;
	this.lookupsModel = this.asst.lookupsModel;
	this.controller = this.asst.controller;
	this.timeout = 1;
	this.searching = false;
	this.currentCountQuery = null;
};

GALHelper.prototype.setup = function () {

};

GALHelper.prototype.toggle = function () {
	Mojo.Dom.toggleClassName(this.toggler, "open");
	Mojo.Dom.toggle(this.listDiv);
};

GALHelper.prototype.reset = function () {
	this.lookupsModel.items = [];
	this.controller.modelChanged(this.lookupsModel);
	Mojo.Dom.removeClassName(this.toggler, "open");
	Mojo.Dom.hide(this.listDiv);
	this.counter.innerHTML = "";
	Mojo.Dom.hide(this.counterContainer);
};

GALHelper.prototype.isOpen = function () {
	return Mojo.Dom.hasClassName(this.toggler, "open");
};

//TODO get better rules for when to search
GALHelper.prototype.doSearch = function (q) {
	this.searching = true;
	this.currentCountQuery = this.asst.controller.serviceRequest("palm://com.palm.mail", {
		method: "queryGAL",
		onSuccess: this.updateCount.bind(this),
		parameters: {
			target: this.asst.filter,
			offset: 0,
			limit: 1
		}
	});
};

GALHelper.prototype.scheduleCountLookup = function () {
	if (!this.searching) {
		this.doSearch();
	} else {
		window.clearTimeout(this.queued);
		this.queued = window.setTimeout(this.doSearch.bind(this), this.timeout);
	}
};

GALHelper.prototype.cancelCountLookup = function () {
	if (this.currentCountQuery) {
		this.currentCountQuery.cancel();
		window.clearTimeout(this.queued);
		this.searching = false;
	}
	this.reset();
};

GALHelper.prototype.updateCount = function (results) {
	this.counter.innerHTML = results;
	Mojo.Dom.show(this.counterContainer);
};

GALHelper.prototype.makeContact = function (rawC) {
	var c, email, number, address;
	
	c = new Contact();
	c.firstName = rawC.firstName;
	c.lastName = rawC.lastName;
	c.jobTitle = rawC.jobTitle;
	c.companyName = rawC.companyName;
	rawC.addressList.forEach(function (addr) {
		if (addr.type === 'EMAIL') {
			email = new Email(addr.value);
			_.extend(email, addr);
			c.emailAddresses.push(email);
		} else if (addr.type === 'PHONE') {
			number = new PhoneNumber(addr.value);
			_.extend(number, addr);
			c.phoneNumbers.push(number);
		} else if (addr.type === 'ADDRESS') {
			address = new Address({
				freeformAddress: addr.value
			});
			_.extend(address, addr);
			c.addresses.push(address);
		}
	});
	return c;
};


//@ sourceURL=contacts.ui/exports.js

/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
	regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Utils, _, Assert, PeoplePickerAssistant, DetailAssistant, PseudoDetailAssistant, ReminderAssistant
	ContactPointPickerAssistant, Contacts, LIB_ROOT, SimloginAssistant */

/**
 * pushPeoplePickerScene
 * 
 * @param {Object} stageController
 * @param {Object} params	{
 *								favoritesOnly: {boolean},
 *								excludeFavorites: {boolean}
 *							}
 */
var pushPeoplePickerScene = exports.pushPeoplePickerScene = function (stageController, launchParams, pushSceneParams) {
	Assert.requireDefined(stageController && stageController.pushScene && _.isFunction(stageController.pushScene), "pushPeoplePicker requires a stageController to be passed");
	
	// extend any extra scene pushing params with the defaults.  The extra pushSceneParams
	// should never override the defaults
	pushSceneParams = _.extend(pushSceneParams || {}, {
		name: "contactsPeoplePickerView",
		assistantConstructor: PeoplePickerAssistant,
		templateRoot: LIB_ROOT,
		sceneTemplate: Utils.getTemplatePath("peoplepicker/peoplepicker-scene")
	});
	
	stageController.pushScene.apply(stageController, [pushSceneParams, launchParams]);
};

/**
 * pushSimLoginScene
 * 
 * @param {Object} stageController
 * @param {Object} attributes
 * @param {Object} model
 */
var pushSimLoginScene = exports.pushSimLoginScene = function (stageController, attributes, model) {
	Assert.requireDefined(stageController && stageController.pushScene && _.isFunction(stageController.pushScene), "pushSimLoginScene requires a stageController to be passed");
	
	// extend any extra scene pushing params with the defaults.  The extra model
	// should never override the defaults
	model = _.extend(model || {}, {
		name: "contactsSimloginView",
		assistantConstructor: SimloginAssistant,
		templateRoot: LIB_ROOT,
		sceneTemplate: Utils.getTemplatePath("simlogin/simlogin-scene")
	});
	
	stageController.pushScene.apply(stageController, [model, attributes]);
};

/**
 * pushPsuedoDetailScene
 * 
 * @param {Object} stageController
 * @param {Object} params
 *					///////////////// DISPLAY ONLY /////////////////
 *					{
 *						personId: {String},			// [Required] The scene will fetch & create the decorated PersonDisplay + linked contacts
 *					}
 *
 *					{
 *						person: {PersonDisplay},	// [Required] decorated PersonDisplay object.  The linked contacts will automatically be fetched if they do not already exist on this object
 *					}
 *
 *					///////////////// ADD NEW ONLY /////////////////
 *					{
 *						contact: {ContactDisplay},	// [Required] The unsaved contact object that will either be added to an existing Person or a new person will be created with this data.
 *					}
 */
var pushPseudoDetailScene = exports.pushPseudoDetailScene = function (stageController, launchParams, pushSceneParams) {
	Assert.requireDefined(stageController && stageController.pushScene && _.isFunction(stageController.pushScene), "pushDetailView requires a stageController to be passed");
	
	var person = launchParams.person,
		contactsFuture;
	if (person && person instanceof Contacts.Person && person.getContacts().length !== person.getContactIds().getLength()) {
		contactsFuture = person.reloadContacts();
	}
	
	// extend any extra scene pushing params with the defaults.  The pushSceneParams
	// should never override the defaults
	pushSceneParams = _.extend(pushSceneParams || {}, {
		name: "pseudoDetailView",
		assistantConstructor: PseudoDetailAssistant,
		templateRoot: LIB_ROOT,
		sceneTemplate: Utils.getTemplatePath("pseudodetail/pseudodetail-scene")
	});
	
	// pushSceneParams - params used by the stageController for how to push the scene
	// params - passed in by the consumer of this library. These params will be passed directly to the scene
	stageController.pushScene.apply(stageController, [pushSceneParams, launchParams, contactsFuture]);
};

/**
 * pushReminderScene
 * 
 * @param {Object} stageController
 * @param {Object} params
 */
var pushReminderScene = exports.pushReminderScene = function (stageController, launchParams, pushSceneParams) {
	Assert.requireDefined(stageController && stageController.pushScene && _.isFunction(stageController.pushScene), "pushReminderScene requires a stageController to be passed");
	
	// extend any extra scene pushing params with the defaults.  The extra pushSceneParams
	// should never override the defaults
	pushSceneParams = _.extend(pushSceneParams || {}, {
		name: "contactsReminderView",
		assistantConstructor: ReminderAssistant,
		templateRoot: LIB_ROOT,
		sceneTemplate: Utils.getTemplatePath("reminder/reminder-scene")
	});
	
	stageController.pushScene.apply(stageController, [pushSceneParams, launchParams]);
};

/**
 * pushContactPointPickerScene
 * 
 * @param {Object} stageController
 * @param {Object} params {
 *		person: {Person},					// Requires either a decorated person or a personId
 *		personId: {string},
 *		type: {string or Array},			// The following constants should be used:	
 *												Contacts.ContactPointTypes.PhoneNumber
 *												Contacts.ContactPointTypes.IMAddress
 *												Contacts.ContactPointTypes.EmailAddress
 *		callback: {Function},				// A function to call when a list item is tapped on
 *		message: {string},					// The message to display in the header
 *		popSceneOnItemSelect: {boolean}		// defaults to true
 *		saveSelectionAsPrimary: {boolean}	// defaults to false. This will save the selected item as the primary for the type that is represents (phone, email, im)
 *	}
 */
var pushContactPointPickerScene = exports.pushContactPointPickerScene = function (stageController, launchParams, pushSceneParams) {
	Assert.requireDefined(stageController && stageController.pushScene && _.isFunction(stageController.pushScene), "pushContactPointPickerScene requires a stageController to be passed");
	
	// extend any extra scene pushing params with the defaults.  The extra pushSceneParams
	// should never override the defaults
	pushSceneParams = _.extend(pushSceneParams || {}, {
		name: "contactPointPicker",
		assistantConstructor: ContactPointPickerAssistant,
		templateRoot: LIB_ROOT,
		sceneTemplate: Utils.getTemplatePath("contactpointpicker/contactpointpicker-scene")
	});
	
	stageController.pushScene.apply(stageController, [pushSceneParams, launchParams]);
};

}