/* Copyright 2009 Palm, Inc.  All rights reserved. */
/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global exports, Assert, Mojo, _, console, Contacts, Utils, EventListenerManager, 
pushPeoplePickerScene, RB, LIB_ROOT, PersonAsyncDecorator, Foundations, Globalization */

var PersonListWidget = exports.PersonListWidget = function () {
	Assert.requireDefined(Mojo, "PersonListWidget: Failed to detect the Mojo framework");
	
	var _sceneController,
		_div,
		_model,
		_uniqueId = "PFLW_" + Date.now(), // DOM elements for this widget that have ids are prefixed with a unique id that is specific to this scene instance
		_stylesheets = [Utils.getStylesheetPath("common.css"), Utils.getStylesheetPath("list.css")],
		_eventListenerManager = new EventListenerManager(),
		_personListScrollPosition,
		_favoriteListScrollPosition,
		_personListDataSource,
		_favoriteListDataSource,
		_personListElement,
		_favoriteListElement,
		_currentView,
		_filterString = "",
		
		
		/**
		* Returns the full id for the specified human readable id
		* @param {String} id
		* @return {String}
		*/
		getWidgetElementId = function (id) {
			return (_uniqueId + id);
		},
		
		/**
		 * Returns the DOM element for the specified human readable id 
		 * @param {String} id
		 * @return {String}
		 */
		getWidgetElement = function (id) {
			return _sceneController.get(getWidgetElementId(id));
		},
		
		transformListItem = function (person) {
			if (!person) {
				return person;
			}
			
			//var start = Date.now();
			person = Contacts.PersonFactory.createPersonDisplayLite(person, _model.sortOrder);
			//console.log("Took " + (Date.now()-start) + "ms to create/transform person object");
			
			// Hide list elements that should be excluded.  Example: when you are manually linking
			// from the detail view, you do not want to show an entry in the list for the person object
			// that you originated from
			if (_model.exclusions && Array.isArray(_model.exclusions)) {
				if (_model.exclusions.indexOf(person._id) !== -1) {
					person.exclude = "exclude";
				}
			}
			
			return person;
		},
		
		dividerLabelCallback = function (person) {
			if (person && !person.exclude) {
				//contacts library - javascript/PersonDisplayLite
				return person.dividerText;
			} else {
				return "";
			}
		},
		
		personDecorator = function (itemModel) {
			var decorations = {},
				displayName = Foundations.StringUtils.escapeHTML(itemModel.displayName);
			
			if (_filterString) {
				// TODO: perf - should cache filter match patterns between invocations
				decorations.displayName = Mojo.PatternMatching.addContactMatchFormatting(displayName, _filterString);
			} else {
				decorations.displayName = displayName;
			}
			
			return decorations;
		},
		
		createAsyncPersonDecorator = function (items, callback) {
			return new PersonAsyncDecorator(items, callback);
		},
				
		setupPersonList = function () {
			var dataSourceAssistant = Contacts.ListWidget.getTypedownDBDataSourceAssistant({
					favoritesOnly: _model.favoritesOnly,
					excludeFavorites: _model.excludeFavorites
				}),
				wrappedDataSourceAssistant = Utils.formatDataSourceAssistant(dataSourceAssistant, transformListItem),
				dividerTemplate;
			
			_personListDataSource = new Mojo.DataSource(wrappedDataSourceAssistant);
			_personListElement = getWidgetElement("person-list");
			_model.personListTapCallback = _model.personListTapCallback || _model.listTapCallback;
			_model.personListDeleteCallback = _model.personListDeleteCallback;
			
			if (_model.sortOrder === Contacts.ListWidget.SortOrder.firstLast || _model.sortOrder === Contacts.ListWidget.SortOrder.lastFirst) {
				dividerTemplate = Utils.getTemplatePath("person-filter-list-widget/group-separator");
			} else {
				dividerTemplate = Utils.getTemplatePath("person-filter-list-widget/multiline-separator");
			}
			
			if(_model.personListDeleteCallback && _.isFunction(_model.personListDeleteCallback))
				var allowDelete=true;
			else
				var allowDelete=false;

			_sceneController.setupWidget(getWidgetElementId("person-list"), {
				templateRoot: LIB_ROOT,
				templates: {
					item: Utils.getTemplatePath("person-filter-list-widget/person-item"),
					empty: Utils.getTemplatePath("person-filter-list-widget/empty")
				},
				uniquenessProperty: "_id",
				dataSource: _personListDataSource,
				dividers: {
					labelCallback: dividerLabelCallback,
					template: dividerTemplate
				},
				decorator: personDecorator,
				asyncDecorator: createAsyncPersonDecorator,
				swipeDelete: allowDelete
			}, {});
			
			if (_model.personListTapCallback && _.isFunction(_model.personListTapCallback)) {
				_eventListenerManager.addListener(_personListElement, Mojo.Event.listTap, _model.personListTapCallback);
			} else {
				console.warn("PersonListWidget (setupPersonList) was set up without a listTapCallback.  Do you really want to do this?");
			}

			if(_model.personListDeleteCallback && _.isFunction(_model.personListDeleteCallback))
				_eventListenerManager.addListener(_personListElement,Mojo.Event.listDelete,_model.personListDeleteCallback);
		},
		
		handleAddFavorite = function (person) {
			person.makeFavorite(true); // pass true so it saves immediately
			// TODO: could add a UI hack to inject this favorite into the list for fast feedback while the DB updates & fires the watch
		},
		
/*		handleAddFavoriteTap = function (event) {
			pushPeoplePickerScene(_sceneController.stageController, {
				excludeFavorites: true,
				iconClass: "icon-add-fav",
				message: RB.$L("Add to Favorites"),
				callback: handleAddFavorite
			});
		},
*/		
		handleFavoriteDelete = function (items, callback) {
			items.forEach(function (item) {
				var person = Contacts.PersonFactory.createPersonDisplay(item);
				person.unfavorite(true);
			});
			
			callback();
		},
		
		setupFavoriteList = function () {
			var dataSourceAssistant = Contacts.ListWidget.getTypedownDBDataSourceAssistant({
					favoritesOnly: true
				}),
				wrappedDataSourceAssistant = Utils.formatDataSourceAssistant(dataSourceAssistant, transformListItem);
			
			_favoriteListDataSource = new Mojo.DataSource(wrappedDataSourceAssistant);
			_favoriteListElement = getWidgetElement("favorite-list");
			_model.favoriteListTapCallback = _model.favoriteListTapCallback || _model.listTapCallback;
			
			//monkey-patch the removeItems so that it doesn't actually delete the person - instead, we just unfavorite it (NOV-115844)
			_favoriteListDataSource.removeItems = handleFavoriteDelete;
			
			_sceneController.setupWidget(getWidgetElementId("favorite-list"), {
				templateRoot: LIB_ROOT,
				templates: {
					item: Utils.getTemplatePath("person-filter-list-widget/person-item")
				},
				uniquenessProperty: "_id",
				dataSource: _favoriteListDataSource,
				decorator: personDecorator,
				asyncDecorator: createAsyncPersonDecorator,
/*				addItem: {
					label: RB.$L("Add Favorite...")
				},
*/
				swipeDelete: {
					deleteText: RB.$L("Remove")
				}
			}, {});
			
			if (_model.favoriteListTapCallback && _.isFunction(_model.favoriteListTapCallback)) {
				_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listTap, _model.favoriteListTapCallback);
			} else {
				console.warn("PersonListWidget (setupFavoriteList) was set up without a listTapCallback.  Do you really want to do this?");
			}
			
//			_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listAdd, handleAddFavoriteTap);
 			_eventListenerManager.addListener(_favoriteListElement, Mojo.Event.listDelete, handleFavoriteDelete);
		},
		
		swapLists = function (elementIdToShow, elementIdToHide, useTransition, state) {
			var elementToShow = getWidgetElement(elementIdToShow),
				elementToHide = getWidgetElement(elementIdToHide),
				transition;
			
			if (useTransition) {
				transition = _sceneController.prepareTransition(Mojo.Transition.crossFade, false);
			}
			
			Mojo.Dom.show(elementToShow);
			Mojo.Dom.hide(elementToHide);
			_sceneController.showWidgetContainer(elementToShow);
			_sceneController.hideWidgetContainer(elementToHide);
			

			if (state) {
				_sceneController.sceneScroller.mojo.setState(state);
			}
			
			
			if (transition) {
				transition.run();
			}
		};
	
	return {
		/**
		* 
		* @param {Object} div
		* @param {Object} sceneController
		* @param {Object} model 
		*					{
		*						mode: {string},
		*						excludeFavorites: {boolean}
		*						listTapCallback: {Function},			// sets the callback for both lists
		*						personListTapCallback: {Function},
		*						favoriteListTapCallback: {Function},
		*						exclusions: {Array} // array of ids,
		*						sortOrder: One of Contacts.ListWidget.SortOrder.XXX
		*					}
		*/
		setup: function (sceneController, div, model) {
			Assert.requireFalse(_sceneController, "PersonListWidget setup() has already been called! Aborting.");
			Assert.require(sceneController, "PersonListWidget requires a sceneController");
			_sceneController = sceneController;
			_div = (div && _.isString(div) ? sceneController.get(div) : div);
			_model = model || {};
			_model.mode = _model.mode || PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST;
			Assert.require(div, "PersonListWidget requires a div element");
			
			if (!_model.sortOrder) {
				Mojo.Log.warning("PersonListWidget: no sort order provided!  Defaulting to Contacts.ListWidget.SortOrder.defaultSortOrder");
				_model.sortOrder = Contacts.ListWidget.SortOrder.defaultSortOrder;
			}
			
			Utils.loadSceneStyles(_sceneController, _stylesheets, _uniqueId);
			
			// inject widget specific HTML into the div supplied by the caller
			var listWidgetModel = {
					uniqueId: _uniqueId
					// if we are only displaying favorites, hide the favorite icon on each list item by
					// setting this class on the outer list widget element.  The CSS on the list item will look for this
					//favoritesListClass: (_model.favoritesOnly ? Contacts.PersonDisplay.FAVORITES_LIST_CLASS : "")
				},
				hasPersonList = false,
				hasFavoriteList = false;
				
			div.innerHTML = Mojo.View.render({object: listWidgetModel, templateRoot: LIB_ROOT, template: Utils.getTemplatePath("person-filter-list-widget/widget-template")});
			
			if (_model.mode === PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST || _model.mode === PersonListWidget.MODE.PERSON_LIST) {
				setupPersonList();
				hasPersonList = true;
			}
			if (_model.mode === PersonListWidget.MODE.PERSON_AND_FAVORITE_LIST || _model.mode === PersonListWidget.MODE.FAVORITE) {
				setupFavoriteList();
				hasFavoriteList = true;
			}
			
			if (_model.defaultView === PersonListWidget.VIEW.PERSON_LIST) {
				this.showPersonList();
			} else if (_model.defaultView === PersonListWidget.VIEW.FAVORITE_LIST) {
				this.showFavoriteList();
			} else {
				// pick a default
				if (hasPersonList) {
					this.showPersonList();
				} else if (hasFavoriteList) {
					this.showFavoriteList();
				}
			}
		},

		getCurrentView: function () {
			return _currentView;
		},
		
		teardown: function () {
			Utils.unloadSceneStyles(_sceneController, _stylesheets, _uniqueId);
			_eventListenerManager.destroyListeners();
		},
		
		showPersonList: function (useTransition) {
			if (_currentView === PersonListWidget.VIEW.PERSON_LIST) {
				return;
			}
			
			_currentView = PersonListWidget.VIEW.PERSON_LIST;
			_favoriteListScrollPosition = _sceneController.sceneScroller.mojo.getState();
			swapLists("person-list-container", "favorite-list-container", useTransition, _personListScrollPosition);
		},
		
		showFavoriteList: function (useTransition) {
			if (_currentView === PersonListWidget.VIEW.FAVORITE_LIST) {
				return;
			}
			_currentView = PersonListWidget.VIEW.FAVORITE_LIST;
			_personListScrollPosition = _sceneController.sceneScroller.mojo.getState();
			
			swapLists("favorite-list-container", "person-list-container", useTransition, _favoriteListScrollPosition);
		},
		
		showAddFavorite: function () {
			pushPeoplePickerScene(_sceneController.stageController, {
				excludeFavorites: true,
				iconClass: "icon-add-fav",
				message: RB.$L("Add to Favorites"),
				callback: handleAddFavorite
			});
		},

		// Filters the visible list
		filter: function (str, setCountFn) {
			var dataSource,
				listElement;
			if (_currentView === PersonListWidget.VIEW.FAVORITE_LIST) {
				dataSource = _favoriteListDataSource;
				listElement = _favoriteListElement;
			} else {
				dataSource = _personListDataSource;
				listElement = _personListElement;
			}
			
			_filterString = str;
			dataSource.setFilterString(str);
			dataSource.getCount(setCountFn);
			listElement.mojo.invalidate();
		},
		
		invalidate: function () {
			//TODO: this can be called when the sort order changes, but we don't update the divider template based on the new sort order!
			
			_favoriteListElement.mojo.invalidate();
			_personListElement.mojo.invalidate();
		}
	};
};

PersonListWidget.MODE = {
	PERSON_LIST: "mode_person_list",
	FAVORITE_LIST: "mode_favorite_list",
	PERSON_AND_FAVORITE_LIST: "mode_person_and_favorite_list"
};

PersonListWidget.VIEW = {
	PERSON_LIST: "view_person_list",
	FAVORITE_LIST: "view_favorite_list"
};
