this._root["__MojoFramework_mojoservice.transport.sync"] = function(MojoLoader, exports, root) {


//@ sourceURL=mojoservice.transport.sync/prologue.js

var IMPORTS = MojoLoader.require(
	{ name: "mojoservice", version: "1.0" }, 
	{ name: "mojoservice.transport", version: "1.0" }, 
	{ name: "foundations", version: "1.0" }, 
	{ name: "foundations.json", version: "1.0" },
	{ name: "foundations.io", version: "1.0" }
);
var MojoService = IMPORTS["mojoservice"];
var Transport = IMPORTS["mojoservice.transport"];
var Foundations = IMPORTS["foundations"];
var FJSON = IMPORTS["foundations.json"];
var IO = IMPORTS["foundations.io"];
var PalmCall = Foundations.Comms.PalmCall;
var AjaxCall = Foundations.Comms.AjaxCall;

var Class = Foundations.Class;
var Future = Foundations.Control.Future;
var DB = Foundations.Data.DB;
var TempDB = Foundations.Data.TempDB;
var SyncStatusManager=Transport.SyncStatusManager;


//@ sourceURL=mojoservice.transport.sync/handlers/synchandler.js

/*global exports, Class, Transport, Future, console */
/*
 * The SyncHandler provide all the infrastructure to synchronize content to the database.
 * A handler is created and given two types, the native database type, and the associated transport
 * database type.
 * The SyncHandler is used in association with the SyncCommand, and provides all the necessary DB operations
 * for that comment.
 */
exports.SyncHandler = function(kinds)
{
	return Class.create(Transport.DbHandler,
	{
		// can return "undefined" for account transport object. Clients will need to detect that
		getAccountTransportObject: function(accountId)
		{
			//console.log(">>>getTransportObject: accountId="+accountId+", kind="+kinds.account.metadata_id);
			return this.find({ from: kinds.account.metadata_id, where: [{ prop: "accountId", op: "=", val: accountId }] }).then(this, function(future)
			{
				var transport = future.result[0];
				future.result = transport;
			});
		},

		putAccountTransportObject: function(transport)
		{
			return this.getAccountTransportObject(transport.accountId).then(this, function(future) {
				transport._rev = future.result._rev;
				future.nest(this.put([ transport ]));
			});
		},

		updateAccountTransportObject: function(transport, updates) {
			return this.getAccountTransportObject(transport.accountId).then(this, function(future) {
				updates._id = future.result._id;
				future.nest(DB.merge([ updates ]));
			});
		},
		
		/*
		 * Get object pairs { local, remote } by the remote id.  The remote id is that used to associate an local
		 * object with its server equivalent.
		 */
		getObjectsByRid: function(rids, name)
		{
			//console.log(">>>getObjectsByRid: kind="+name);
			function getMetadataObjectsThenMergeThem(object_kind, metadata_kind) {
				return function(future) {
					var objs=[];
					var rmap={};
					var f = this.find({ from: metadata_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId},{ prop: "rid", op: "=", val: rids }] }).then(this, function(future) {
						if (future.result.length) {
							objs = future.result;
						}
						return objs;
					}).then(this, function(future) {
						// create mapping from localId->remoteId
						var lids=[];
						for (var i=0; i < objs.length; i++) {
							var o = objs[i];
							rmap[o.lid]=o.rid;
							lids[i]=o.lid;
						}
						// find local objects matching remote objects
						return this.find({ from: object_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId},{ prop: "_id", op: "=", val: lids }] });
					}).then(this, function(future) {
						// merge remoteIds from metadata objects
						var locals = future.result;
						var updates = [];
						for (var i=0; i < locals.length; i++) {
							var l = locals[i];
							var id = l._id;
							if (!l.remoteId) {
								var remoteId = rmap[id];
								if (remoteId) {
									updates.push({
										_id: id,
										remoteId: remoteId
									});
								}
							}
						}
						return this.merge(updates);
					}).then(this, function(future) {
						// delete and purge the metadata objects, we won't be needing them again
						return this.del({ from: metadata_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId},{ prop: "rid", op: "=", val: rids }] }, true);
					});
					return f;
				};
			}
			function getObjects(object_kind) {
				//console.log(">>>getObjects("+object_kind+")");
				return function(future) {
					var rmap = {};
					rids.forEach(function(rid)
					{
						rmap[rid] = null;
					});
					this.find({ from: object_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId},{ prop: "remoteId", op: "=", val: rids }] }).then(this, function(future2) 
					{
						if (future2.exception) {
							console.error("getObjects: exception raised, but I'm ignoring it:"+future2.exception);
							future.result = future.result;
							return;
						}
						var results = [];
						future2.result.forEach(function(result)
						{
							// mapping remote->local
							results.push({ local: result, remote: { remoteId: result.remoteId } });
							delete rmap[result.remoteId];
						});
						// Create mappings for new objects
						for (var key in rmap)
						{
							if (rmap.hasOwnProperty(key)) {
								// mapping remote->new object
								results.push({ local: {_kind: object_kind, remoteId: key}, remote: { remoteId: key } });
							}
						}
						future.result = results;
					});
				};
			}
			var object_kind = kinds.objects[name].id;
			var metadata_kind = kinds.objects[name].metadata_id;
			var f = new Future(true);
			if (metadata_kind) {
				f.then(this, getMetadataObjectsThenMergeThem(object_kind, metadata_kind));
			}
			f.then(this, getObjects(object_kind));
			return f;
		},
		/*
		 * Get local objects which have changed since the given 'rev'
		 */
		getChangedObjects: function(rev, name)
		{
			//console.log(">>>getChangedObjects rev="+rev+", kind="+JSON.stringify(name));
			function getChanged(object_kind) {
				return function(future) {
					console.log(">>>getChanged object_kind: "+object_kind+", _rev: "+rev);
					future.nest(this.find({ from: object_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId},{ prop: "_rev", op: ">", val: rev }], incDel: true}).then(this, function(future2)
					{
						if (future2.exception) {
							console.error("getObjects: exception raised, but I'm ignoring it:"+future2.exception);
							future2.result = future2.result;
							return;
						}
						var results = [];
						future2.result.forEach(function(result)
						{
							if (result.remoteId) {
								// mapping local->remote
								results.push({ local: result, remote: {remoteId: result.remoteId} });
							} else {
								// Create mappings for new objects
								results.push({ local: result, remote: {} });
							}
						});
						results.sort(function(a, b) {
							if (a.local._rev > b.local._rev) {
								return 1;
							} else if (a.local._rev < b.local._rev) {
								return -1;
							} else {
								return 0;
							}
						});
						future2.result = results;
					}));
				};
			}			
			var f = new Future(true);
			var object_kind = kinds.objects[name].id;
			f.then(this, getChanged(object_kind));
			return f;
		},

		getLatestRev: function(name) {
			var object_kind = kinds.objects[name].id;
			var f = this.find({ from: object_kind, where: [{ prop: "accountId", op: "=", val: this.command.client.clientId}], orderBy: "_rev", desc: true, limit: 1});
			f.then(function (future)
			{
				var result = future.getResult();
				if (result && result.length)
				{
					future.result = result[0]._rev;
				}
				else
				{
					future.result = 0;
				}
			});
			return f;
		},

		/*
		 * Put the object pairs { local, transport } to the database.
		 * Delete pairs which are marked as 'delete'.
		 * returns an object containing the {id, rev} pairs for all modified objects, and [id, id] for deleted records
		 * {
		 *   put: [{id: id1, rev: rev1}...],
		     deleted: [id1, id2, id3]
		 * }
		 */
		putObjects: function(objects)
		{
			return new Future().now(this, function(future)
			{
				//console.log(">>>putObjects ");
				// Calculate number of ids we need
				var count = 0;
				objects.forEach(function(obj)
				{
					if (obj.operation == "save" && !obj.local._id)
					{
						count++;
					}
				});
				//console.log(">>count="+count);
				// Get the ids
				this.reserveIds(count).then(this, function(future2)
				{
					var ids = future2.result;
					var i = 0;
					var saved = [];
					var deleted = [];
					var accountId = this.command.client.clientId;
					// Create an array of items we need to save and delete, and allocate new ids
					// where necessary. 
					objects.forEach(function(obj)
					{
						switch (obj.operation)
						{
							case "save":
								saved.push(obj.local);
								if (!obj.local._id)
								{
									obj.local._id = ids[i++];
									obj.local.accountId = accountId;
								}
								break;
								
							case "delete":
								deleted.push(obj.local._id);
								break;
						}
					});
					this.del(deleted, false).then(this, function(future2)
					{
						future2.result=future2.result; // Check for errors
						this.put(saved).then(function(future2)
						{
							future.result = {
								"put": future2.result,
								"deleted": deleted
							};
						});
					});
				});
			});
		},
		
		/*
		 * Put transport objects to the database.
		 */
		putTransportObjects: function(objs, kindName)
		{
			return this.put(objs);
		},
		
		/*
		 * Create a new account.
		 */
		createAccount: function()
		{
			//console.log(">>> createAccount");
			var accountId = this.command.client.clientId;
			// note that the modnum is set to 0 here, but gets reset in enableAccount
			return this.put([{ _kind: kinds.account.metadata_id, accountId: accountId, modnum: 0, syncKey: {}, extras: {}, initialSync: true }]).then(function(future)
			{
				future.result = true;
			});
		},
		
		/*
		 * Enable an account.
		 */
		enableAccount: function(extras)
		{
			//console.log(">>> enableAccount");
			var accountId = this.command.client.clientId;
			// find the account transport object
			return this.getAccountTransportObject(accountId).then(this, function(future)
			{
				var result = future.result;
				if (result) {
					//this is the normal case - just pass the result on
					future.result = result;
				} else {
					//if we're doing this after a restore, and the account transport object wasn't backed up, 
					//then we need to create one and fetch it.
					console.warn("enableAccount: unable to retrieve account object to enable account, creating one");
					return this.createAccount().then(this, function(innerFuture)
					{
						innerFuture.getResult();
						return this.getAccountTransportObject(accountId);
					});
				}
			}).then(this, function(future)
			{
				//now, either way, we should have an account transport object
				var result = future.result;
				if (result) {
					var newFields = {
						_id: result._id,
						extras: extras
					};
					if (result.modnum === 0) {  // this is an initial Sync situation: enable/disable, or newly created account
						newFields.syncKey = null;
						newFields.initialSync = true;
					}
					future.nest(this.merge([newFields]).then(this, function(future2) {
						//reset modnum to match current DB revision
						newFields.modnum = future2.result[0].rev;
						console.log(">>> Reset modnum to "+newFields.modnum);
					
						if (result.modnum === 0) {  // this is an initial Sync situation: enable/disable, or newly created account
							//set the syncKey to {} instead of the null above so that it's an empty object in the db
							//disableAccount clears is, but if the transport object was backed up and restored, it won't 
							//be empty, so we clear it again here to be safe.
							newFields.syncKey = {};
						}
					
						future2.nest(this.merge([newFields]));
					}).then(function(future) {
						return (result.modnum === 0);
					}));
				} else {
					console.warn("enableAccount: unable to create account object to enable account");
					future.result = false;
				}
			});
		},
		
		/*
		 * Disable an account.
		 */
		disableAccount: function()
		{
			//console.log(">>> disableAccount");
			var accountId = this.command.client.clientId;
			var extras = null;
			// find the account
			function delKind(kind) {
				return function(future) {
					//console.log("delKind: "+kind);
					var f2 = this.del({ from: kind, where: [ { prop: "accountId", op: "=", val: accountId } ] }, false);
					f2.then(function() {
						if (f2.exception) {
							console.error("delKind: exception raised, but I'm ignoring it:"+f2.exception);
						}
						f2.result=true;
					});
					future.nest(f2);
				};
			}
			var f = this.getAccountTransportObject(accountId).then(this, function(future)
			{
				var result = future.result;
				if (result) {
					extras = result.extras;
					var updatedTransportObject = {
						_kind: result._kind,
						_id: result._id,
						_rev: result._rev,
						accountId: result.accountId,
						modnum: 0, // will be set correctly in enableAccount
						syncKey: {},
						extras: {}
					};
					future.nest(this.put([updatedTransportObject]).then(function(future2) {
						future2.getResult();
						future2.result = true;
					}));
				} else {
					console.warn("disableAccount: unable to retrieve account object to disable it");
					future.result = false;
				}
			});
			
			// Remove the sync object kinds
			var syncOrder = kinds.syncOrder;
			for (var i=0; i<syncOrder.length; i++) {
				var syncObject = syncOrder[i];
				var object_kind = kinds.objects[syncObject].id;
				f.then(this, delKind(object_kind));
			}
			f.then(function(future)
			{
				//console.log("setting final result");
				future.result = extras;
			});
			return f;
		},
		
		/*
		 * Delete all objects associated with this account.
		 */
		deleteAccount: function()
		{
			//console.log(">>> deleteAccount");
			var accountId = this.command.client.clientId;
			return this.del({ from: kinds.account.metadata_id, where: [ { prop: "accountId", op: "=", val: accountId } ] }, false);
		},
		
		getObjectKinds: function() 
		{
			return kinds;
		}
	});
};

//@ sourceURL=mojoservice.transport.sync/commands/synccommand.js

/*global console, Class, TempDB, Transport, Future, Foundations, exports, require: true, IMPORTS,
file, PalmCall, SyncStatusManager, AjaxCall */

/*
 * The SyncCommand provide all the handling to synchronize content between a server and the client.
 * This class should be extended for use with specific servers and specific content types, but is general
 * enough to containt most (if not all) of the heavy lifting necessary in providing two-way content sync.
 */
var SyncCommand = exports.SyncCommand = Class.create(Transport.Command,
{
	commandTimeout: 60 * 60, // 60 minutes

	/*
	 * Returns a function which will transform between local and remote formats.
	 * The name describes which direction should be provided - currently supports
	 * local2remote and remote2local.  If local2remote is not supported, return undefined
	 * and this will become a readonly sync.
	 * The transformation function takes the form 'bool function(to, from)' and returns a
	 * defined value (of some sort) if the transform of from makes a change in 'to'.
	 */
	getTransformer: function(name, kindName)
	{
		throw new Error("No getTransformer function");
	},

	/*
	 * Returns the unique identifier for that object.  This is used to track syncing of the local and remote
	 * copies.
	 */
	getRemoteId: function(obj, kindName)
	{
		throw new Error("No getRemoteId function");
	},

	/*
	 * Returns true if the objects has been deleted from the server (ie. this is a tombstone).
	 */
	isDeleted: function(obj, kindName)
	{
		throw new Error("No isDeleted function");
	},

	/*
	 * Returns a set of remote changes from the server.
	 */
	getRemoteChanges: function(state, kindName)
	{
		throw new Error("No remote object function");
	},

	/*
	 * Given a set of remote ids, returns a set of remote objects matching those ids.
	 */
	getRemoteMatches: function(ids, kindName)
	{
		throw new Error("No remote matches function");
	},

	/*
	 * Put a set of remote objects to the server.  Each object has an operation property
	 * which is either 'save' or 'delete', depending on how the objects should be put
	 * onto the server.
	 */
	putRemoteObjects: function(objects, kindName)
	{
		throw new Error("No remote put function");
	},

	/*
	 * Create an 'empty' remote objects which can then have the local content
	 * transformed into.
	 */
	getNewRemoteObject: function(kindName)
	{
		throw new Error("No new remote object function");
	},

	/*
	 * Return an array of "identifiers" to identify object types for synchronization
	 * and what order to sync them in
	 * This will normally be an array of strings, relating to the getSyncObjects function:
	 * [ "contactset", "contact" ]
	 */
	getSyncOrder: function() {
		throw new Error("No getSyncOrder function");
	},

	/*
	 * Return an array of "kind objects" to identify object types for synchronization
	 * This will normally be an object with property names as returned from getSyncOrder, with structure like this:
	 * {
	 *   contact: {
	 *	 id: com.palm.contact.google:1
	 *	 metadata_id: com.palm.contact.google.transport:1
	 *	 NOTE: metadata_id is deprecated. All data should be stored in the "id" type
	 *   }
	 * }
	 */
	getSyncObjects: function() {
		throw new Error("No getSyncObjects function");
	},

	/*
	 * Return the ID string for the capability (e.g., CALENDAR, CONTACTS, etc.)
	 * supported by the sync engine as specified in the account template (e.g.,
	 * com.palm.calendar.google, com.palm.contacts.google, etc.).  This is used
	 * to provide automatic sync notification support.
	 */
	getCapabilityProviderId: function() {
	},

	preSaveModify: function() {
		console.log("synccommand: preSaveModify()");
		return new Future([]);
	},

	postPutRemoteModify: function() {
		console.log("synccomand: postPutRemoteModify()");
		return new Future([]);
	},

	initialize: function()
	{
	},

	run: function(future)
	{
		this._future = future;
		var serviceAssistant = this.controller.service.assistant;
		serviceAssistant._syncInProgress = serviceAssistant._syncInProgress || {};

		if (serviceAssistant._syncInProgress[this.client.clientId])
		{
			console.log("Sync activity already in progress, ignoring sync request");
			future.result={returnValue:true, result:"sync already in progress"};
		}
		else if (this.controller.args.$activity &&  this.controller.args.$activity.trigger &&  this.controller.args.$activity.trigger.returnValue===false) {
			// error during triggered activity - probaby a bad watch
			var response = this.controller.args.$activity.trigger;
			console.error("Error with triggered activity:");
			console.error("error in trigger: "+JSON.stringify(response));
			future.result={returnValue:false, result:"error in trigger: "+JSON.stringify(response)};
		}
		else // start the sync machinery
		{
			serviceAssistant._syncInProgress[this.client.clientId]=true;
			new Foundations.Control.FSM(this);
			var self = this;
			this._gotReply = function(future)
			{
				self.event("gotReply", future);
			};
		}
	},

	"yield": function() {
		console.warn("Yield requested. Will stop syncing at next checkpoint");
		this._yieldRequested = true;
	},

	getPeriodicSyncActivityName: function() {
		var name = "Periodic Sync:"+this.controller.service.name + ":" + this.client.clientId; //TODO: clean this up here and in EnableAccountCommand
		return name;
	},

	getPeriodicSyncActivity: function() {
		var name = this.getPeriodicSyncActivityName();
		var details = PalmCall.call("palm://com.palm.activitymanager", "getDetails", {"activityName": name}).then(this, function(future) {
			// got it - return details
			future.result = future.result.activity;
		},
		function(future) {
			// error - create activity
			var error = future.exception;
			if (error.errorCode === 2) {
				console.log("Periodic Sync Activity not found, re-creating it");
			} else {
				console.error("Error getting details for Sync Activity, re-creating it: " + error);
			}
			var inner;
			if (this.client.getSyncInterval && typeof this.client.getSyncInterval === 'function') {
				inner = this.client.getSyncInterval();
			} else {
				console.error("=== No getSyncInterval function in client for "+this.controller.service.name +" ===");
				console.error("=== Default sync interval is 24 hours ===");
				inner = new Future("24h");
			}
			future.nest(inner).then(this, function(future) {
				//ToDo: merge this with implementation in EnableAccoutCommand
				var interval;
				if (future.exception) {
					console.error("Error in client.getSyncInterval, assuming syncInterval 24h");
					interval="24h";
				} 
				else {
					interval = future.result;
				}
				var requiresInternet;
				var requirements;
				if (this.client.requiresInternet && typeof this.client.requiresInternet === 'function') {
					try {
						requiresInternet = this.client.requiresInternet();
					}
					catch (_) {
						console.error("client error in requiresInternet");
						requiresInternet=true;
					}
				} else {
					console.error("=== No requiresInternet function in client for "+this.controller.service.name +" ===");
					console.error("=== Default answer is 'true' - internet is required ===");
					requiresInternet = true;
				}
				requirements = (requiresInternet) ? { internetConfidence: "fair" } : undefined;
				var args = { accountId: this.client.clientId };
				var activity = new Foundations.Control.Activity(name, "Periodic Sync", true)
					.setScheduleInterval(interval)
					.setUserInitiated(false)
					.setExplicit(true)
					.setPersist(true)
					.setReplace(true)
					.setRequirements(requirements)
					.setCallback("palm://" + this.controller.service.name + "/"+this.controller.config.name, args);
				return activity.start();
			});
		});
		return details;
	},

	complete: function(activity) {
		console.log("Completing activity "+activity.name);
		var syncActivity;
		var details = this.getPeriodicSyncActivity().then(function(future) {
			var restart=false;
			syncActivity = future.result;
			if (activity._activityId === syncActivity.activityId) {
				console.log("Periodic sync. Restarting activity");
				restart=true;
			} else {
				console.log("Not periodic sync. Completing activity");
			}
			return activity.complete(restart);
		}).then(function(future) {
			console.log("Complete succeeded, result = "+JSON.stringify(future.result));
			future.result=true;
		},
		function(future) {
			console.log("Complete FAILED, exception = "+JSON.stringify(future.exception));
			future.result=false;
		}).then(this, function(future) {
			if (future.result) {
				// TODO: Set up one of these for each synced kind...
				if (this._future.result && this._local2remoteTransformer) { // if we can up-sync, set up a watch to kick of sync on edit
					var rev = this.client.transport.modnum;
					var name = "SyncOnEdit:"+this.controller.service.name + ":" + this.client.clientId; //TODO: clean this up here and in EnableAccountCommand
					var requiresInternet;
					var requirements;
					if (this.client.requiresInternet && typeof this.client.requiresInternet === 'function') {
						requiresInternet = this.client.requiresInternet();
					} else {
						console.error("=== No requiresInternet function in client for "+this.controller.service.name +" ===");
						console.error("=== Default answer is 'true' - internet is required ===");
						requiresInternet = true;
					}
					requirements = (requiresInternet) ? { internetConfidence: "fair" } : undefined;
					var queryParams = {
						"query":{
							"from":this._kind,
							"where":[
								{"prop":"accountId", "op":"=", "val":this.client.transport.accountId},
								{"prop":"_rev", "op":">", "val": rev}
							],
							incDel: true
						},
						"subscribe": true
					};
					var args = { accountId: this.client.clientId };
					var activity = new Foundations.Control.Activity(name, "Sync On Edit", true)
						.setUserInitiated(false)
						.setExplicit(true)
						.setPersist(true)
						.setReplace(true)
						.setRequirements(requirements)
						.setTrigger("fired", "palm://com.palm.db/watch", queryParams)
						.setCallback("palm://" + this.controller.service.name + "/"+this.controller.config.name, args);
					return activity.start();
				}
			}
			future.result=true;
		});
		return details;
	},

	__start:
	{
		__enter: function()
		{
			try {
				this._syncOrder = this.getSyncOrder();
			} catch (e) {
				console.error("error in getSyncOrder: "+e._stack?e.stack:e.toString());
				this._error=e;
				return "error";
			}
			try {
				this._syncObjects =this.getSyncObjects();
			} catch (e2) {
				console.error("error in getSyncObjects: "+e2._stack?e2.stack:e2.toString());
				this._error=e2;
				return "error";
			}
			this._upsyncedSomething=false;
			this._syncCount=0;
			console.log(">>>syncOrder = "+JSON.stringify(this._syncOrder));
			this._kindIndex=0;

			var capability;
			try {
				capability = this.getCapabilityProviderId();
			} catch(e3) {
				console.error("error in getCapabilityProviderId: "+e3._stack?e3.stack:e3.toString());
				this._error=e3;
				return "error";
			}
			this.syncStatusMgr = new SyncStatusManager(this.client.clientId, capability, this.controller.service.name);
			this.syncStatusMgr.clearSyncStatus().then(this, function (future) {
				future.getResult();
				if (this.client.transport && this.client.transport.initialSync) {
					return this.syncStatusMgr.setInitialSyncStatus();
				} else {
					return this.syncStatusMgr.setIncrementalSyncStatus();
				}
			}).then(this._gotReply);
		},

		gotReply: function(join) {
			join.getResult();
			return "getFirstRemoteChanges";
		}
	},

	getFirstRemoteChanges:
	{
		__enter: function()
		{
			this._kindName=this._syncOrder[this._kindIndex];
			console.log(">>> kindName="+this._kindName);
			this._kind=this._syncObjects[this._kindName].id;
			console.log(">>> kind="+JSON.stringify(this._kind));

			try {
				this._remote2localTransformer = this.getTransformer("remote2local", this._kindName);
			} catch (e) {
				console.error("error in getTransformer: "+e._stack?e.stack:e.toString());
				this._error=e;
				return "error";
			}
			console.log(">>> remote2localTransformer="+ this._remote2localTransformer);

			try {
				this._local2remoteTransformer = this.getTransformer("local2remote", this._kindName);
			} catch (e2) {
				console.error("error in getTransformer: "+e2._stack?e2.stack:e2.toString());
				this._error=e2;
				return "error";
			}
			if (!this._remote2localTransformer) {
				console.log(">>>No remote2local transformer defined for "+this._kindName+"! Going to next kind");
				return "nextType";
			}
			this._localChangeMap = {};
			this._serverDeleteMap = {};
			this._remoteState = "first";
			return "getMoreRemoteChanges";
		}
	},

	getMoreRemoteChanges:
	{
		__enter: function()
		{
			try {
				this.getRemoteChanges(this._remoteState, this._kindName).then(this._gotReply);
			} catch (e) {
				console.error("error in getRemoteChanges: "+e._stack?e.stack:e.toString());
				this._error=e;
				return "error";
			}
		},

		gotReply: function(join)
		{
			try
			{
				this._remoteState = join.result.more ? "more" : "last";
				this._remoteChanges = join.result.entries;
				//console.log(">>>this._remoteChanges:"+JSON.stringify(this._remoteChanges));
				return "getLocalMatches";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	getLocalMatches:
	{
		__enter: function()
		{
			var self = this;
			var batch = {};
			this._remoteChanges.forEach(function(change)
			{
				try {
					batch[self.getRemoteId(change, self._kindName)] = change;
				} catch (e) {
					console.error("error in getRemoteId: "+e._stack?e.stack:e.toString());
					this._error=e;
					return "error";
				}
			});
			this._remoteChanges = batch;
			//console.log(">>>this._remoteChanges:"+JSON.stringify(this._remoteChanges));
			this.handler.getObjectsByRid(Object.keys(batch), this._kindName).then(this._gotReply);
		},

		gotReply: function(join)
		{
			var map = this._remoteChanges;
			this._remoteChanges = join.result;
			//console.log(">>>this._remoteChanges:"+JSON.stringify(this._remoteChanges));
			join.result.forEach(function(result)
			{
				result.remote = map[result.remote.remoteId];
			});
			return "mergeRemoteChanges";
		}
	},

	mergeRemoteChanges:
	{
		__enter: function()
		{
			var transformer = this._remote2localTransformer;
			var wb = [];
			var self = this;
			//console.log(">>>this._remoteChanges:"+JSON.stringify(this._remoteChanges));
			this._remoteChanges.forEach(function(match)
			{
				var isDeleted;
				try {
					isDeleted = self.isDeleted(match.remote, self._kindName);
				} catch (e) {
					console.error("error in isDeleted: "+e._stack?e.stack:e.toString());
					this._error=e;
					return "error";
				}
				if (isDeleted)
				{
					if (match.local._id)
					{
						match.operation = "delete";
						wb.push(match);
					}
					// else this object was deleted remotely, but there is no matching local object, so we ignore it.
				}
				else
				{
					// merge changes from remote to local objecs
					var t;
					try {
						t = transformer(match.local, match.remote);
					} catch (e2) {
						console.error("error in remote2localtransformer: "+e2._stack?e2.stack:e2.toString());
						this._error=e2;
						return "error";
					}
					if (t)
					{
						// transformer returns true if anything changed
						match.operation = "save";
						wb.push(match);
					}
				}
			});
			this._remoteChanges=undefined;
			this._localWriteback = wb;

			return "preSaveModifyStep";
		}
	},

	preSaveModifyStep:
	{
		__enter: function()
		{
			console.log("preSaveModify");
			// modify local objects before pushing to database
			try {
				this.preSaveModify(this._localWriteback).then(this._gotReply);
			} catch (e) {
				console.error("error in remote2localtransformer: "+e._stack?e.stack:e.toString());
				this._error=e;
				return "error";
			}
		},

		gotReply: function(future)
		{
			try
			{
				future.getResult();
				return "writeLocalChanges";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	writeLocalChanges:
	{
		__enter: function()
		{
			//console.log(">>>__enter");
			this.handler.putObjects(this._localWriteback).then(this._gotReply);
		},

		gotReply: function(join)
		{
			this._localWriteback = undefined;
			//console.log(">>>gotReply join="+JSON.stringify(join));
			try
			{
				// Update local change map - used to avoid sending our immediate changes back to the server
				var puts = join.result.put;
				var map = this._localChangeMap;
				puts.forEach(function(result)
				{
					map[result.id] = result.rev;
				});
				// Update serverDeleteMap - used to avoid sending server deletes right back to the server on up-sync
				var deletes = join.result.deleted;
				map = this._serverDeleteMap;
				deletes.forEach(function(result)
				{
					map[result] = true; // add deleted records to map
				});

				// If we're done then update the account, otherwise get the next batch
				if (this._remoteState == "last")
				{
					if (this._local2remoteTransformer)
					{
						return "checkpointDownSync";
					}
					else
					{
						return "nextType";
					}
				}
				else
				{
					return "getMoreRemoteChanges";
				}
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	// Down-sync complete. Save the syncKey, so we don't repeat all that work if anything goes wrong during upsync
	checkpointDownSync:
	{
		__enter: function()
		{
			console.log("saving syncKey");
			this.handler.updateAccountTransportObject(this.client.transport, {syncKey: this.client.transport.syncKey}).then(this._gotReply);
		},
		gotReply: function(future)
		{
			// NOV-119682
			// if yield was requested, end the sync now. Otherwise, continue to up-sync
			if (this._yieldRequested) {
				console.warn("Yield: Bailing out after down-sync");
				return "updateAccount";
			}
			else {
				return "getLocalChanges";
			}
		}
	},

	getLocalChanges:
	{
		__enter: function()
		{
			this.handler.getChangedObjects(this.client.transport.modnum, this._kindName).then(this._gotReply);
		},

		gotReply: function(join)
		{
			try
			{
				// Filter out any remote2local changes we just made and find highest revision
				var map = this._localChangeMap;
				var delMap = this._serverDeleteMap;
				var rev = this.client.transport.modnum;
				var changes = [];
				join.result.forEach(function(result)
				{
					// if changed locally, and not deleted on server
					// TODO: This doesn't actually work quite correctly. See NOV-117942 for details
					if (map[result.local._id] !== result.local._rev && !delMap[result.local._id])
					{
						changes.push(result);
					}
					if (result.local._rev > rev)
					{
						rev = result.local._rev;
					}
				});
				for (var id in map) {
					if (rev < map[id]) {
						rev = map[id];
					}
				}
				this._latestRev = rev;
				this._localChanges = changes;
				return "getFirstRemoteMatches";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	getFirstRemoteMatches:
	{
		__enter: function()
		{
			var rids = [];
			var self = this;
			var localChangesToSyncUp = [];
			this._localChanges.forEach(function(change, index)
			{
				if (change.local.remoteId)
				{
					//this is a local modify or delete on a contact that exists remotely
					localChangesToSyncUp.push(change);
					rids.push(change.local.remoteId);
				}
				else if (!change.local._del)
				{
					//this is a local add
					localChangesToSyncUp.push(change);
					try {
						change.remote = self.getNewRemoteObject(self._kindName);
					} catch (e) {
						console.error("error in getNewRemoteObject: "+e._stack?e.stack:e.toString());
						this._error=e;
						return "error";
					}
				}
				// else this is a local delete that happened on a brand-new local contact that was
				// never synced to the server. so we just drop the change
			});
			//now replace this._localChanges with localChangesToSyncUp so that we drop the
			//objects that were added locally and then deleted before getting synced up
			this._localChanges = localChangesToSyncUp;
			this._rids = rids;
			try {
				this.getRemoteMatches(rids, this._kindName).then(this._gotReply);
			} catch (e) {
				console.error("error in getRemoteMatches: "+e._stack?e.stack:e.toString());
				this._error=e;
				return "error";
			}
		},

		gotReply: function(join)
		{
			try
			{
				var matches = this._localChanges;
				var pos=0;
				for (var i=0; i < join.result.length; i++)
				{
					var rid = this._rids[i];
					while (matches[pos].local.remoteId != rid)
					{
						pos++;
					}
					matches[pos++].remote = join.result[i];
				}
				return "mergeLocalChanges";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	mergeLocalChanges:
	{
		__enter: function()
		{
			var transformer = this._local2remoteTransformer;
			var wb = [];
			this._localChanges.forEach(function(match)
			{
				if (match.local._del)
				{
					match.operation = "delete";
					wb.push(match);
				}
				else
				{
					var t;
					try {
						t = transformer(match.remote, match.local);
					} catch (e) {
						console.error("error in local2remotetransformer: "+e._stack?e.stack:e.toString());
						this._error=e;
						return "error";
					}
					if (t)
					{
						match.operation = "save";
						wb.push(match);
					}
				}
			});
			this._remoteWriteback = wb;

			return "writeRemoteChanges";
		}
	},

	writeRemoteChanges:
	{
		__enter: function()
		{
			// TODO: remove this when NOV-117942 is fixed properly
			if (this._remoteWriteback.length > 0) {
				this._upsyncedSomething = true;
			}
			// TODO: make this configurable, for transports that can do all-or-nothing upsync
			var one_change = this._remoteWriteback.shift();
			if (!one_change) {
				return "nextType";
			}
			this._batch = [one_change];
			try {
				this.putRemoteObjects(this._batch, this._kindName).then(this._gotReply);
			} catch (_) {
				console.error("error in putRemoteObjects: "+_._stack?_.stack:_.toString());
				this._error=_;
				return "error";
			}
		},

		gotReply: function(join)
		{
			try
			{
				// copy any changed remoteIds from putRemoteObjects into the local objects
				var results = join.result;
				var len = results.length;
				var transports = [];
				for (var i = 0; i < len; i++)
				{
					var result = results[i];
					var local = this._batch[i].local;
					if (result !== local.remoteId)
					{
						local.remoteId = result;
					}
					transports.push(local);
				}
				this._batch.transport = transports;
				return "writeRemoteTransportChanges";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	writeRemoteTransportChanges:
	{
		__enter: function()
		{
			// save all the local objects
			this.handler.putTransportObjects(this._batch.transport, this._kindName).then(this._gotReply);
		},

		gotReply: function(join)
		{
			try
			{
				// copy any changed revs from the put into the local objects
				var results = join.result;
				var len = results.length;
				for (var i = 0; i < len; i++)
				{
					var result = results[i];
					var local = this._batch[i].local;
					if (result.id !== local._id) {
						throw new Error("ID mismatch from putTransportObjects()");
					} else {
						if (local._rev && (result.rev !== local._rev)) {
							local._rev=result.rev;
						}
					}
				}
				this._batch.transport = undefined;
				join.result = true;
				return "getPostPutRemoteChanges";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	getPostPutRemoteChanges:
	{
		__enter: function()
		{
			// Give the client an opportunity to modify local objects based on
			// results from putting those objects to the server (e.g., etags)
			try {
				this.postPutRemoteModify(this._batch, this._kindName).then(this._gotReply);
			} catch (_) {
				console.error("error in postPutRemoteModify: "+_._stack?_.stack:_.toString());
				this._error=_;
				return "error";
			}
		},

		gotReply: function(join)
		{
			try
			{
				var results = join.result;
				var len = results.length;
				if (len)
				{
					for (var i = 0; i < len; ++i)
					{
						results[i] = {
							local: results[i],
							operation: "save"
						};
					}
					this._batch = results;
					return "writePostPutRemoteChanges";
				}
				else
				{
					return "getPostPutRemoteModnum";
				}
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	writePostPutRemoteChanges:
	{
		__enter: function()
		{
			console.log(">>> writePostPutRemoteChanges: " + JSON.stringify(this._remoteWriteback));
			this.handler.putObjects(this._batch).then(this._gotReply);
		},

		gotReply: function(join)
		{
			var results,
				len,
				result,
				local,
				i,
				j,
				wbLen;
			try
			{
				results = join.getResult().put;
				//set the changed revs for the local objects that are on the write array which have not yet been
				//written but have been modified in the meanwhile so that there won't be conflicting revs
				//at the moment that they will be written into the DB
				len = results.length;
				for (i = 0;i < len;i++)
				{
					result = results[i];
					wbLen = this._remoteWriteback.length;
					for (j = 0; j < wbLen; j++) {
						local = this._remoteWriteback[j].local;
						if (result.id === local._id && local._rev && (result.rev !== local._rev)) {
							local._rev = result.rev;
							break;
						}
					}
				}
				return "getPostPutRemoteModnum";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	getPostPutRemoteModnum:
	{
		__enter: function()
		{
			try
			{
				var latestRev = this._latestRev;
				console.log(">>> getPostPutRemoteModnum: cur: " + this.client.transport.modnum + ", latest: " + JSON.stringify(latestRev));
				if (latestRev > this.client.transport.modnum)
				{
					this.client.transport.modnum = latestRev;
					this.handler.updateAccountTransportObject(this.client.transport, {modnum: this.client.transport.modnum});
				}
				return "nextType";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	nextType:
	{
		__enter: function()
		{
			if (this._remoteWriteback && this._remoteWriteback.length > 0) {
				return "writeRemoteChanges";
			}
			if (this._latestRev !== undefined && this._latestRev > this.client.transport.modnum) {
				return "getPostPutRemoteModnum";
			}
			console.log(">>> this._kindIndex = "+this._kindIndex+", this._kinds.syncOrder.length-1="+(this._syncOrder.length-1));
			if (this._kindIndex < this._syncOrder.length-1) {
				this._kindIndex++;
				return "getFirstRemoteChanges";
			}
			return "updateAccount";
		}
	},

	updateAccount:
	{
		__enter: function()
		{
			this.handler.updateAccountTransportObject(this.client.transport, {initialSync: false, syncKey: this.client.transport.syncKey}).then(this._gotReply);
		},

		gotReply: function(join)
		{
			try
			{
				return join.result && "success";
			}
			catch (_)
			{
				console.log(_.stack);
				this._error = _;
				return "error";
			}
		}
	},

	success:
	{
		__enter: function()
		{
			var serviceAssistant = this.controller.service.assistant;
			if (this._upsyncedSomething && this._syncCount < 2) {
				/* restart current sync */
				console.log("Upsync ocurred, restarting sync");
				this._syncCount++;
				this._upsyncedSomething = false;
				return "getFirstRemoteChanges";
			}
			serviceAssistant._syncInProgress[this.client.clientId]=false;
			this.syncStatusMgr.clearSyncStatus().then(this, function (join) {
				join.getResult();
				console.log("synccommand(success): __enter");
				this._future.result = {};
			});
		}
	},

	error:
	{
		__enter: function()
		{
			var serviceAssistant = this.controller.service.assistant;
			serviceAssistant._syncInProgress[this.client.clientId]=false;

			//TODO: NOV-111365: before we clear the sync status, we need to check to see if it's a recoverable network error,
			//		in which case we should retry the sync (based on a retry count provided by the sync engine)

			this.syncStatusMgr.clearSyncStatus().then(this, function (join) {
				join.getResult();

				//this is the list of cases where we need to notify the user of the error
				if (this._error instanceof Transport.TransportError) {
					this.syncStatusMgr.setErrorCondition(this._error);
				}

				//TODO: in some of these cases (e.g. Transport.AuthenticationError), we should also stop the scheduled syncs
				//		until the user corrects the error

				//whether or not we notify the user of the error, we need to propagate it upwards
				this._future.exception = this._error || new Error("Unspecified error");
			});
		}
	}
});

/*
 * Static function that checks if the photo object already exists in the filecache,
 * if not it will fetch the photo and insert it into the filecache using the function
 * passed in to fetch the resource
 * Returns the local path to the photo
 */
SyncCommand.fetchPhoto = function( photo, headers )
{
	var future = new Future(),
		cacheInsertFuture,
		haveCanceledSubscription = false,
		fs,
		urlObject;

	future.now(function () {
		if (typeof require === 'undefined') {
			require = IMPORTS.require;
		}
		fs = require('fs');
		var url = require('url');

		console.log("synccommand.fetchPhoto() : " + JSON.stringify( photo ) );

		// Check if a filepath already exists
		if (photo.localPath) {
			try
			{
				var exists = fs.openSync(photo.localPath, "r");
				if (exists)
				{
					fs.closeSync(exists);
					console.log("fetchPhoto: photo already exists: " + photo.localPath);
					return new Future({
						skippedInsert: true,
						path: photo.localPath
					});
				}
			}
			catch (e)
			{
				console.log("File does not exist: " + photo.localPath);
			}
		}

		urlObject = url.parse(photo.value);
		var name = urlObject.pathname.substring(urlObject.pathname.lastIndexOf("/") + 1) || "unknown";
		// This size is an estimate which will be resized accordingly once the photo has been downloaded
		//TODO: this should really reference Contacts.PersonPhotos.BIG_PHOTO_FILECACHE_SIZE and Contacts.PersonPhotos.LIST_PHOTO_FILECACHE_SIZE
		var size = (photo.type === "type_big") ? 34816 : 8192;

		cacheInsertFuture = PalmCall.call("palm://com.palm.filecache/", "InsertCacheObject", {
			typeName: "contactphoto",
			fileName: name,
			size: size,
			subscribe: true
		});

		return cacheInsertFuture;
	});

	// Copy the image to the filecache
	future.then(function () {
		var result = future.result,
			path = future.result.pathName;

		if (result.skippedInsert) {
			return result.path;
		}

		console.log("fetchPhoto: filecache object inserted at: " + path);

		// open a file for writing. We're using openSync here so as not to have to
		// deal with asynchronous I/O
		var file = fs.openSync(path, "w");
		// total bytes transferred
		var count = 0;

		// set up the options for this request - we have custom headers and an onData callback
		var options = {};
		options.headers = headers;
		options.onData = function onDataCallback(chunk) {
			//console.log("chunk.length="+chunk.length);
			// write chunk to file '0' is the offset in the buffer, and
			// chunk.length writes the whole thing. By default, it writes nothing...
			fs.writeSync(file, chunk, 0, chunk.length);
			count += chunk.length;
		};

		//now make the request
		var ajaxCallFuture = AjaxCall.call(AjaxCall.RequestMethod.GET, urlObject.href, "", options);

		ajaxCallFuture.then(function () {
			var status = ajaxCallFuture.result.status;

			console.log("File finished downloading - (" + count + "): " + path);

			// close the file we had open
			fs.closeSync(file);

			// If it failed to download, expire the file and throw an error
			if (status !== 200) {
				var expireFuture = PalmCall.call("palm://com.palm.filecache", "ExpireCacheObject", {
						pathName: path
				});

				expireFuture.then(function () {
					expireFuture.getResult();
					throw new Error(ajaxCallFuture.result.responseText);
				});

				return expireFuture;
			}

			var resizeFuture = PalmCall.call("palm://com.palm.filecache/", "ResizeCacheObject", {
				pathName: path,
				newSize: count
			});
			resizeFuture.then(function () {
				var result = resizeFuture.result;

				//cancel the subscription we had open with the filecache so that it will save the object
				PalmCall.cancel(cacheInsertFuture);
				haveCanceledSubscription = true;

				// Now that the file exists on the local filesystem, set the path in the photo object
				// so it can make use of it
				photo.localPath = path;

				//also return the path to the caller via the future chain
				return path;
			});
			return resizeFuture;
		});

		return ajaxCallFuture;
	});

	future.then(function () {
		try {
			console.log("fetchPhoto: image copied to file cache: " + JSON.stringify(future.result) );
			return future.result;
		} catch(e) {
			console.log("fetchPhoto failed: " + e.stack);
			throw e;
		} finally {
			//just in case we didn't cancel it above, cancel it now
			if (!haveCanceledSubscription) {
				PalmCall.cancel(cacheInsertFuture);
			}
		}
	});

	return future;
};


//@ sourceURL=mojoservice.transport.sync/commands/refetchphotocommand.js

/*global console, DB, Future, exports:true, Class, Transport, IO, PalmCall */

/*
 * Fetch the photos using this transport.
 */
exports.RefetchPhotoCommand = Class.create(Transport.Command,
{
    fetchPhoto: function (photo) {
        console.log("RefetchPhotoCommand: 'fetchPhoto' has not been implemented by the engine");
    },

    getKind: function () {
		throw new Error("No getKind function");
    },

    /*
     * Lookup the contact in the contact database
     * loop through the contact's photos searching for the specified photo
     * if the photo is found, fetch it to the filecache and update the localPath
     * merge the new path back into the contacts database
     * Arguments to this service call are:
            accountId: accountId,
            contactId: contactId,
            photoId: photoId
     */
	run: function (param)
	{
        var self = this,
            photo = null,
            photos = null,
            future = null,
            args = this.controller.args;

        if (!args.contactId || !args.photoId)
        {
            console.log("RefetchPhotoCommand: Missing required parameter: " + JSON.stringify(args));
            return new Future({});
        }

        console.log("RefetchPhotoCommand: searching for contactId: " + args.contactId);

        // get the contact using the contactId
		//TODO: make this a DB.get instead of a DB.find
        future = DB.find({
            from: this.getKind(),
            where: [ {"op": "=", "prop": "_id", "val": args.contactId} ]
        });

        future.then(function ()
        {
            var i,
                result = future.result;

            if (!result || !result.results || result.results.length === 0) 
            {
                throw new Error("RefetchPhotoCommand: no results searching for contact");
            }
            if (result.results.length > 1) 
            {
                throw new Error("RefetchPhotoCommand: too many results while searching for contact");
            }

            photos = result.results[0].photos;

            // get the specified photo from the photo array
            for (i = 0; i < photos.length; i++)
            {
                if (photos[i]._id === args.photoId)
                {
                    photo = photos[i];
                    console.log("RefetchPhotoCommand: Found desired photo = " + JSON.stringify(photo));
                    return self.fetchPhoto(photo);
                }
            }

            throw new Error("Did not find photo: " + args.photoId + " for contact: " + args.contactId);
        });

        future.then(function ()
        {
            photo.localPath = future.result;

            console.log("RefetchPhotoCommand: new photo path is: " + JSON.stringify(future.result));

            var query = {
                from: self.getKind(),
                where: [
                    {"op": "=", "prop": "_id", "val": args.contactId}
                ]
            };

            return DB.merge(query, {photos: photos});
        });

        future.then(function ()
        {
            try {
                console.log("RefetchPhotoCommand: DB merge returned: " + JSON.stringify(future.result));
                return future.result;
            } catch (e) {
                console.log("RefetchPhotoCommand: DB merge failed: " + e);
                throw e;
            }
        });

        param.result = future.result;
	}
});


//@ sourceURL=mojoservice.transport.sync/commands/createaccountcommand.js

exports.CreateAccountCommand = Class.create(Transport.Command,
{
	run: function(result)
	{
		var future = new Future(true);
		result.nest(future.then(this,
		[
			function()
			{
				return this.handler.createAccount();
			},
			function()
			{
				// Confirm account was created
				future.result = {};
			}
		]));
	}
});


//@ sourceURL=mojoservice.transport.sync/commands/enabledaccountcommand.js

/*global Class, Transport, Future, Foundations, exports, PalmCall, console */
exports.EnabledAccountCommand = Class.create(Transport.Command,
{
	commandTimeout: 3600,
	run: function(result)
	{
		var future = new Future(true),
			args = this.controller.args;
	
		if(args.enabled) { // account is being enabled
			var interval;
			var requiresInternet;
			var requirements;
			if (this.client.requiresInternet && typeof this.client.requiresInternet === 'function') {
				try {
					requiresInternet = this.client.requiresInternet();
				}
				catch (e) {
					console.error("client error in requiresInternet");
					requiresInternet=true;
				}
			} else {
				console.error("=== No requiresInternet function in client for "+this.controller.service.name +" ===");
				console.error("=== Default answer is 'true' - internet is required ===");
				requiresInternet = true;
			}
			requirements = (requiresInternet) ? { internetConfidence: "fair" } : undefined;
			var command = "sync"; //TODO: Shouldn't the command name be retrieved from the config?
			var args = { accountId: this.client.clientId };
			
			result.nest(future.then(this,
			[
				function()
				{
					if (this.client.getSyncInterval && typeof this.client.getSyncInterval === 'function') {
						return this.client.getSyncInterval();
					} else {
						console.error("=== No getSyncInterval function in client for "+this.controller.service.name +" ===");
						console.error("=== Default sync interval is 24 hours ===");
						return new Future("24h");
					}
				},
				function()
				{
					var interval;
					if (future.exception) {
						console.error("Error in client.getSyncInterval, assuming syncInterval 24h");
						interval="24h";
					} 
					else {
						interval = future.result;
					}
					var activity = new Foundations.Control.Activity("Periodic Sync:"+this.controller.service.name + ":" + this.client.clientId, "Periodic Sync", true)
						.setScheduleInterval(interval)
						.setUserInitiated(false)
						.setExplicit(true)
						.setPersist(true)
						.setReplace(true)
						.setRequirements(requirements)
						.setCallback("palm://" + this.controller.service.name + "/"+command, args);
					return activity.start();
				},
				function(future)
				{
					var activityId = future.result.activityId;
					return this.handler.enableAccount({ syncActivityId: activityId }); //TODO: remove syncActivityId here - it's no longer used
				},
				function(future)
				{
					var initialSync=future.result;
					// Post command for the initial sync
					if (initialSync) {
						var activity = new Foundations.Control.Activity("Initial Sync:"+this.controller.service.name + ":" + this.client.clientId, "Initial Sync", true)
							.setUserInitiated(true)
							.setExplicit(true)
							.setPersist(true)
							.setReplace(true)
							.setCallback("palm://" + this.controller.service.name + "/"+command, args);
						return activity.start();
					} else {
						return true;
					}
				}
			]));
		} else {	// account is being disabled
			var syncActivity;
			var serviceAssistant = this.controller.service.assistant;
			var clientId=this.client.clientId;
			serviceAssistant._syncInProgress = serviceAssistant._syncInProgress || {};
			
			if (serviceAssistant._syncInProgress[clientId]) {
				result.result={"returnValue":false, "errorText":"Sync in progress", "errorCode":"BUSY"};
				return;
			}
			serviceAssistant._syncInProgress[clientId]=true;
			
			result.nest(future.then(this,
			[
				function(future) {
					var name = "Periodic Sync:"+this.controller.service.name + ":" + this.client.clientId; // TODO: merge this with code in SyncCommand
					// get the activityId from ActivityManager
					return PalmCall.call("palm://com.palm.activitymanager", "getDetails", {"activityName":name}); 
				},
				function(future)
				{
					// Cancel the periodic sync activity
					console.log("cancelling periodic sync");
					if (future.exception) {
						console.error("ignoring exception from Activity Manager");
						return {returnValue: true};
					}
					var activity = future.result.activity;
					return PalmCall.call("palm://com.palm.activitymanager", "cancel", {"activityId":activity.activityId}); 
				},
				function(future) {
					if (future.exception) {
						console.error("ignoring exception from Activity Manager");
					}
					var name = "SyncOnEdit:"+this.controller.service.name + ":" + this.client.clientId; // TODO: merge this with code in SyncCommand
					// get the activityId from ActivityManager
					return PalmCall.call("palm://com.palm.activitymanager", "getDetails", {"activityName":name}); 
				},
				function(future)
				{
					// Cancel the triggered sync activity
					if (future.exception) {
						console.error("ignoring exception from Activity Manager");
						return {returnValue: true};
					}
					console.log("cancelling triggered sync");
					var activity = future.result.activity;
					return PalmCall.call("palm://com.palm.activitymanager", "cancel", {"activityId":activity.activityId}); 
				},
				function()
				{
					return this.handler.disableAccount();
				},
				function(future)
				{
					if (future.exception) {
						console.error("ignoring exception from this.handler.disableAccount()");
					}
					serviceAssistant._syncInProgress[clientId]=false;
					return true;
				}
			]));
		}	
	}
});


//@ sourceURL=mojoservice.transport.sync/commands/deleteaccountcommand.js

exports.DeleteAccountCommand = Class.create(Transport.Command,
{
	run: function(result)
	{
		var future = new Future(true);
		result.nest(future.then(this,
		[
			function()
			{
				return this.handler.deleteAccount();
			},
			function()
			{
				future.result = {};
			}
		]));
	}
});

//@ sourceURL=mojoservice.transport.sync/clients/authsyncclient.js

/*global exports, console, Transport, Class */
exports.AuthSyncClient = Class.create(Transport.Client,
{
	__start:
	{
		__enter: function()
		{
			return "restart";
		}
	},
	
	restart:
	{
		__enter: function()
		{
			// Load the account transport object
			this.handlerFactory.getHandler(undefined).getAccountTransportObject(this.clientId).then(this, function(future)
			{
				this.event("gotReply", future);
			});
		},
		
		gotReply: function(future)
		{
				this.transport = future.result;
			if (!this.transport) {
				console.info("MojoService.AuthSyncClient(restart): no transport object");
				// If we fail to get the transport, we leave it unset.
				// This will only happen if we've not yet created it, and the currently executing command will
				// be onCreate
			}
			return "unauthorized";
		}
	},
	
	unauthorized:
	{
		__enter: function()
		{
			this.dispatchCommand();
		},
		
		__any: function(command)
		{
			this.queueCommand(Transport.Priority.NORMAL, command);
			
			// If we support authorization, then issue the authenticate command
			if (this.getAuthenticateCommandDescription)
			{
				this.runCommand(Transport.Priority.IMMEDIATE, this.createCommand(this.getAuthenticateCommandDescription(), command));
			}
			// Otherwise we always authorizaed
			else
			{
				return "authorized";
			}
		},
		
		authenticate: function(command)
		{
			console.log("authenticate");
			this.activateCommand(command);
			return "inAuthorize";
		}
	},
	
	inAuthorize:
	{
		__commandComplete: function(cmd)
		{
			console.log("__commandComplete", cmd.name);
			try
			{
				cmd.controller.future.result;
				return "authorized";
			}
			catch (e)
			{
				console.warn("MojoService.AuthSyncClient(inAuthorize): error thrown"+e);
			}
			return "restart";
		}
	},
	
	authorized:
	{
		__enter: function()
		{
			this.dispatchCommand();
		},
		
		checkCredentials: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForComplete";
		},
		
		sync: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForComplete";
		},
		
		__any: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForComplete";
		},
		
		onCreate: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForCreateComplete";
		},
		
		onEnabled: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForEnabledComplete";
		},
		
		onDelete: function(cmd)
		{
			this.activateCommand(cmd);
			return "waitForComplete";
		}
	},
	
	waitForComplete:
	{
		__commandComplete: function(cmd)
		{
			console.log("authsyncclient waitForComplete command: ");
			
			try
			{
				cmd.controller.future.result;
				return "authorized";
			}
			catch (e)
			{
				console.warn("MojoService.AuthSyncClient(waitForComplete): error thrown: "+e);
			}
			return "restart";
		}
	},

	waitForCreateComplete:
	{
		__commandComplete: function(cmd)
		{
			console.log("authsyncclient waitForCreateComplete command: ");
			return "restart";
		}
	},

	waitForEnabledComplete:
	{
		__commandComplete: function(cmd)
		{
			console.log("authsyncclient waitForEnabledComplete command: ");
			return "restart";
		}
	},
	
	inResource:
	{
		__commandComplete: function(cmd)
		{
			console.log("authsyncclient inResource command: ");
			return "authorized";
		}
	}
});

}