/*jslint white: true, onevar: true, undef: true, eqeqeq: true, plusplus: true, bitwise: true, 
regexp: true, newcap: true, immed: true, nomen: false, maxerr: 500 */
/*global google, PhoneNumber, EmailAddress, IMAddress, Address, Url, Organization, Relation, _, console, Globalization */

var GoogleContactHandler = {};

GoogleContactHandler._typeLookup = function (typeMap, inType, inTypeIsLocal) {
	var outType = _.detect(typeMap, function (typeMapItem) {
		if (inTypeIsLocal) {
			return typeMapItem.local === inType;
		} else {
			return typeMapItem.remote === inType;
		}
	});
	
	outType = outType || typeMap.defaultType;
	
	return (inTypeIsLocal) ? outType.remote : outType.local;
};

/*
 * Phone Number Types
 */
GoogleContactHandler._phoneTypeMap = [{
	local: PhoneNumber.TYPE.MOBILE,
	remote: google.gdata.PhoneNumber.REL_MOBILE
}, {
	local: PhoneNumber.TYPE.MOBILE,
	remote: google.gdata.PhoneNumber.REL_WORK_MOBILE
}, {
	local: PhoneNumber.TYPE.HOME,
	remote: google.gdata.PhoneNumber.REL_HOME
}, {
	local: PhoneNumber.TYPE.WORK,
	remote: google.gdata.PhoneNumber.REL_WORK
}, {
	local: PhoneNumber.TYPE.MAIN,
	remote: google.gdata.PhoneNumber.REL_MAIN
}, {
	local: PhoneNumber.TYPE.PERSONAL_FAX,
	remote: google.gdata.PhoneNumber.REL_FAX
}, {
	local: PhoneNumber.TYPE.PERSONAL_FAX,
	remote: google.gdata.PhoneNumber.REL_HOME_FAX
}, {
	local: PhoneNumber.TYPE.PERSONAL_FAX,
	remote: google.gdata.PhoneNumber.REL_OTHER_FAX
}, {
	local: PhoneNumber.TYPE.WORK_FAX,
	remote: google.gdata.PhoneNumber.REL_WORK_FAX
}, {
	local: PhoneNumber.TYPE.PAGER,
	remote: google.gdata.PhoneNumber.REL_PAGER
}, {
	local: PhoneNumber.TYPE.PAGER,
	remote: google.gdata.PhoneNumber.REL_WORK_PAGER
}, {
	local: PhoneNumber.TYPE.ASSISTANT,
	remote: google.gdata.PhoneNumber.REL_ASSISTANT
}, {
	local: PhoneNumber.TYPE.CAR,
	remote: google.gdata.PhoneNumber.REL_CAR
}, {
	local: PhoneNumber.TYPE.COMPANY,
	remote: google.gdata.PhoneNumber.REL_COMPANY_MAIN
}, {
	local: PhoneNumber.TYPE.RADIO,
	remote: google.gdata.PhoneNumber.REL_RADIO
}, {
	local: PhoneNumber.TYPE.OTHER,
	remote: google.gdata.PhoneNumber.REL_OTHER
}];
GoogleContactHandler._phoneTypeMap.defaultType = {
	local: PhoneNumber.TYPE.OTHER,
	remote: google.gdata.PhoneNumber.REL_OTHER
};

/*
 * Email Address Types
 */
GoogleContactHandler._emailTypeMap = [{
	local: EmailAddress.TYPE.HOME,
	remote: google.gdata.Email.REL_HOME
}, {
	local: EmailAddress.TYPE.WORK,
	remote: google.gdata.Email.REL_WORK
}, {
	local: EmailAddress.TYPE.OTHER,
	remote: google.gdata.Email.REL_OTHER
}];
GoogleContactHandler._emailTypeMap.defaultType = {
	local: EmailAddress.TYPE.OTHER,
	remote: google.gdata.Email.REL_OTHER
};

/*
 * IM Address Label Types
 */
GoogleContactHandler._imLabelMap = [{
	local: IMAddress.LABEL.HOME,
	remote: google.gdata.Im.REL_HOME
}, {
	local: IMAddress.LABEL.WORK,
	remote: google.gdata.Im.REL_WORK
}, {
	local: IMAddress.LABEL.OTHER,
	remote: google.gdata.Im.REL_OTHER
}];
GoogleContactHandler._imLabelMap.defaultType = {
	local: IMAddress.LABEL.OTHER,
	remote: google.gdata.Im.REL_OTHER
};

/*
 * IM Address Service Name Types
 */
GoogleContactHandler._imTypeMap = [{
	local: IMAddress.TYPE.AIM,
	remote: google.gdata.Im.PROTOCOL_AIM
}, {
	local: IMAddress.TYPE.GTALK,
	remote: google.gdata.Im.PROTOCOL_GOOGLE_TALK
}, {
	local: IMAddress.TYPE.ICQ,
	remote: google.gdata.Im.PROTOCOL_ICQ
}, {
	local: IMAddress.TYPE.JABBER,
	remote: google.gdata.Im.PROTOCOL_JABBER
}, {
	local: IMAddress.TYPE.MSN,
	remote: google.gdata.Im.PROTOCOL_MSN
}, {
	local: IMAddress.TYPE.QQ,
	remote: google.gdata.Im.PROTOCOL_QQ
}, {
	local: IMAddress.TYPE.SKYPE,
	remote: google.gdata.Im.PROTOCOL_SKYPE
}, {
	local: IMAddress.TYPE.YAHOO,
	remote: google.gdata.Im.PROTOCOL_YAHOO
}, {
	local: IMAddress.TYPE.DEFAULT,
	remote: ""
}];
GoogleContactHandler._imTypeMap.defaultType = {
	local: IMAddress.TYPE.DEFAULT,
	remote: ""
};

/*
 * Postal Address Types
 */
GoogleContactHandler._addressTypeMap = [{
	local: Address.TYPE.HOME,
	remote: google.gdata.StructuredPostalAddress.REL_HOME
}, {
	local: Address.TYPE.WORK,
	remote: google.gdata.StructuredPostalAddress.REL_WORK
}, {
	local: Address.TYPE.OTHER,
	remote: google.gdata.StructuredPostalAddress.REL_OTHER
}];
GoogleContactHandler._addressTypeMap.defaultType = {
	local: Address.TYPE.OTHER,
	remote: google.gdata.StructuredPostalAddress.REL_OTHER
};

/*
 * URL Types
 */
GoogleContactHandler._urlTypeMap = [{
	local: Url.TYPE.HOME,
	remote: google.gdata.contacts.Website.REL_HOME
}, {
	local: Url.TYPE.WORK,
	remote: google.gdata.contacts.Website.REL_WORK
}, {
	local: Url.TYPE.HOMEPAGE,
	remote: google.gdata.contacts.Website.REL_HOME_PAGE
}, {
	local: Url.TYPE.BLOG,
	remote: google.gdata.contacts.Website.REL_BLOG
}, {
	local: Url.TYPE.FTP,
	remote: google.gdata.contacts.Website.REL_FTP
}, {
	local: Url.TYPE.PROFILE,
	remote: google.gdata.contacts.Website.REL_PROFILE
}, {
	local: Url.TYPE.OTHER,
	remote: google.gdata.contacts.Website.REL_OTHER
}];
GoogleContactHandler._urlTypeMap.defaultType = {
	local: Url.TYPE.OTHER,
	remote: google.gdata.contacts.Website.REL_OTHER
};

/*
 * Organization Types
 */
GoogleContactHandler._organizationTypeMap = [{
	local: Organization.TYPE.WORK,
	remote: google.gdata.Organization.REL_WORK
}, {
	local: Organization.TYPE.OTHER,
	remote: google.gdata.Organization.REL_OTHER
}];
GoogleContactHandler._organizationTypeMap.defaultType = {
	local: Organization.TYPE.WORK,
	remote: google.gdata.Organization.REL_WORK
};

/*
 * Relation Types
 */
GoogleContactHandler._relationTypeMap = [{
	local: Relation.TYPE.ASSISTANT,
	remote: google.gdata.contacts.Relation.REL_ASSISTANT
}, {
	local: Relation.TYPE.BROTHER,
	remote: google.gdata.contacts.Relation.REL_BROTHER
}, {
	local: Relation.TYPE.CHILD,
	remote: google.gdata.contacts.Relation.REL_CHILD
}, {
	local: Relation.TYPE.DOMESTIC_PARTNER,
	remote: google.gdata.contacts.Relation.REL_DOMESTIC_PARTNER
}, {
	local: Relation.TYPE.FATHER,
	remote: google.gdata.contacts.Relation.REL_FATHER
}, {
	local: Relation.TYPE.FRIEND,
	remote: google.gdata.contacts.Relation.REL_FRIEND
}, {
	local: Relation.TYPE.MANAGER,
	remote: google.gdata.contacts.Relation.REL_MANAGER
}, {
	local: Relation.TYPE.MOTHER,
	remote: google.gdata.contacts.Relation.REL_MOTHER
}, {
	local: Relation.TYPE.PARENT,
	remote: google.gdata.contacts.Relation.REL_PARENT
}, {
	local: Relation.TYPE.PARTNER,
	remote: google.gdata.contacts.Relation.REL_PARTNER
}, {
	local: Relation.TYPE.REFERRED_BY,
	remote: google.gdata.contacts.Relation.REL_REFERRED_BY
}, {
	local: Relation.TYPE.RELATIVE,
	remote: google.gdata.contacts.Relation.REL_RELATIVE
}, {
	local: Relation.TYPE.SISTER,
	remote: google.gdata.contacts.Relation.REL_SISTER
}, {
	local: Relation.TYPE.SPOUSE,
	remote: google.gdata.contacts.Relation.REL_SPOUSE
}, {
	local: Relation.TYPE.OTHER,
	remote: ""
}];
GoogleContactHandler._relationTypeMap.defaultType = {
	local: Relation.TYPE.OTHER,
	remote: ""
};

/*
 * Gender Types
 */
GoogleContactHandler._genderTypeMap = [{
	local: "female",
	remote: google.gdata.contacts.Gender.VALUE_FEMALE
}, {
	local: "male",
	remote: google.gdata.contacts.Gender.VALUE_MALE
}, {
	local: "undisclosed",
	remote: ""
}];
GoogleContactHandler._genderTypeMap.defaultType = {
	local: "undisclosed",
	remote: ""
};

GoogleContactHandler.getRemoteId = function (remoteObject) {
	if (remoteObject) {
		//TODO: should this use getSelfLink instead?
		var idObj = remoteObject.getId();
		if (idObj) {
			return idObj.getValue() || undefined;
		}
	}
	
	return undefined;
};

GoogleContactHandler.copyFieldsToLocal = function (remoteContact, localContact) {
	var deleted;
	
	localContact.id = GoogleContactHandler.getRemoteId(remoteContact);
	
	deleted = remoteContact.getDeleted();
	if (deleted) {
		localContact.isDeleted = true;
	}
	
	GoogleContactHandler._copyNameToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyNicknameToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyGenderToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyBirthdayToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyAnniversaryToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyNoteToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyCustomFieldsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyPhoneNumbersToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyEmailsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyImsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyAddressesToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyUrlsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyOrganizationsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyRelationsToLocal(localContact, remoteContact);
	
	GoogleContactHandler._copyPhotoToLocal(localContact, remoteContact);
		
	return localContact;
};

String.prototype.rtrim = function () {
	return this.replace(/[ \t\r\n]+$/g, "");
}
GoogleContactHandler._ltrim = function (object) {
	for (var name in object) {
		var value = object[name];
		switch (typeof value) {
		case 'string':
			object[name] = value.rtrim();
			break;

		case 'object':
			this._ltrim(object[name]);
			break;
		}
	}
}

GoogleContactHandler.copyFieldsToRemote = function (localContact, remoteContact) {
	this._ltrim(localContact);
	GoogleContactHandler._copyNameToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyNicknameToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyGenderToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyBirthdayToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyAnniversaryToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyNoteToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyCustomFieldsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyPhoneNumbersToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyEmailsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyImsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyAddressesToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyUrlsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyOrganizationsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyRelationsToRemote(localContact, remoteContact);
	
	GoogleContactHandler._copyPhotoToRemote(localContact, remoteContact);
	
	return remoteContact;
};




/*
 * "Private" helper methods
 */


/*
 * Copy to local methods
 */

GoogleContactHandler._copyNameToLocal = function (localContact, remoteContact) {
	var nameObj,
		prefixObj,
		givenNameObj,
		additionalNameObj,
		familyNameObj,
		suffixObj;
	
	nameObj = remoteContact.getName();
	localContact.name = {};
	if (nameObj) {
		prefixObj = nameObj.getNamePrefix();
		if (prefixObj) {
			localContact.name.honorificPrefix = prefixObj.getValue();
		}
		givenNameObj = nameObj.getGivenName();
		if (givenNameObj) {
			localContact.name.givenName = givenNameObj.getValue();
		}
		additionalNameObj = nameObj.getAdditionalName();
		if (additionalNameObj) {
			localContact.name.middleName = additionalNameObj.getValue();
		}
		familyNameObj = nameObj.getFamilyName();
		if (familyNameObj) {
			localContact.name.familyName = familyNameObj.getValue();
		}
		suffixObj = nameObj.getNameSuffix();
		if (suffixObj) {
			localContact.name.honorificSuffix = suffixObj.getValue();
		}
	}
};

GoogleContactHandler._copyNicknameToLocal = function (localContact, remoteContact) {
	var nicknameObj = remoteContact.getNickname();
	if (nicknameObj) {
		localContact.nickname = nicknameObj.getValue();
	}
};

GoogleContactHandler._copyGenderToLocal = function (localContact, remoteContact) {
	var genderObj = remoteContact.getGender();
	if (genderObj) {
		localContact.gender = GoogleContactHandler._typeLookup(GoogleContactHandler._genderTypeMap, genderObj.getValue(), false);
	}
};

GoogleContactHandler._copyBirthdayToLocal = function (localContact, remoteContact) {
	var birthdayObj = remoteContact.getBirthday();
	if (birthdayObj) {
		localContact.birthday = birthdayObj.getWhen();
		if (localContact.birthday && localContact.birthday.charAt(0) === '-') {
			localContact.birthday = "0000" + localContact.birthday.substring(1, localContact.birthday.length);
		}
	}
};

GoogleContactHandler._copyAnniversaryToLocal = function (localContact, remoteContact) {
	var events = remoteContact.getEvents(),
		curEventObj,
		i,
		whenObj;
	
	if (events && Array.isArray(events)) {
		//for all the other events, grab the first one that claims to be an anniversary and store that.
		for (i = 0; i < events.length; i += 1) {
			curEventObj = events[i];
			if (curEventObj.getRel() === google.gdata.contacts.Event.REL_ANNIVERSARY) {
				whenObj = curEventObj.getWhen();
				if (whenObj) {
					localContact.anniversary = google.gdata.DateTime.toIso8601(whenObj.getStartTime());
				}
				break;
			}
		}
	}
};

GoogleContactHandler._copyNoteToLocal = function (localContact, remoteContact) {
	var noteObj = remoteContact.getContent();
	if (noteObj) {
		localContact.note = noteObj.getText();
	}
};

GoogleContactHandler._copyCustomFieldsToLocal = function (localContact, remoteContact) {
	var customFields = remoteContact.getUserDefinedFields();
	localContact.customFields = [];
	if (customFields && Array.isArray(customFields)) {
		customFields.forEach(function (customField) {
			localContact.customFields.push({
				//this should use customField.getKey(), but that doesn't seem to retrieve the right property
				name: customField.key,
				value: customField.getValue()
			});
		});
	}
};

GoogleContactHandler._copyPhoneNumbersToLocal = function (localContact, remoteContact) {
	var phoneNumbers = remoteContact.getPhoneNumbers();
	localContact.phoneNumbers = [];
	if (phoneNumbers && Array.isArray(phoneNumbers)) {
		phoneNumbers.forEach(function (phoneNumber) {
			localContact.phoneNumbers.push({
				value: phoneNumber.getValue(),
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._phoneTypeMap, phoneNumber.getRel(), false),
				primary: phoneNumber.getPrimary()
			});
		});
	}
};

GoogleContactHandler._copyEmailsToLocal = function (localContact, remoteContact) {
	var emails = remoteContact.getEmailAddresses();
	localContact.emails = [];
	if (emails && Array.isArray(emails)) {
		emails.forEach(function (email) {
			localContact.emails.push({
				value: email.getAddress(),
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._emailTypeMap, email.getRel(), false),
				primary: email.getPrimary()
			});
		});
	}
};

GoogleContactHandler._copyImsToLocal = function (localContact, remoteContact) {
	var ims = remoteContact.getImAddresses();
	localContact.ims = [];
	if (ims && Array.isArray(ims)) {
		ims.forEach(function (im) {
			localContact.ims.push({
				value: im.getAddress(),
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._imTypeMap, im.getProtocol(), false),
				label: GoogleContactHandler._typeLookup(GoogleContactHandler._imLabelMap, im.getRel(), false),
				primary: im.getPrimary()
			});
		});
	}
};

GoogleContactHandler._copyAddressesToLocal = function (localContact, remoteContact) {
	var addresses = remoteContact.getStructuredPostalAddresses();
	localContact.addresses = [];
	if (addresses && Array.isArray(addresses)) {
		addresses.forEach(function (address) {
			var streetAddress,
				locality,
				region,
				postalCode,
				country,
				formattedAddressObj,
				formattedAddressValue,
				parsedAddress;
			
			streetAddress = address.getStreet() && address.getStreet().getValue();
			locality = address.getCity() && address.getCity().getValue();
			region = address.getRegion() && address.getRegion().getValue();
			postalCode = address.getPostcode() && address.getPostcode().getValue();
			country = address.getCountry() && address.getCountry().getValue();
			if (streetAddress || locality || region || postalCode || country) {
				//if we have any of these fields, let's use them
				localContact.addresses.push({
					streetAddress: streetAddress,
					locality: locality,
					region: region,
					postalCode: postalCode,
					country: country,
					type: GoogleContactHandler._typeLookup(GoogleContactHandler._addressTypeMap, address.getRel(), false),
					primary: address.getPrimary()
				});
			} else {
				//since we didn't have the parsed fields, let's just use the fulltext version and parse it ourselves
				formattedAddressObj = address.getFormattedAddress();
				if (formattedAddressObj) {
					formattedAddressValue = formattedAddressObj.getValue();
					if (formattedAddressValue) {
						parsedAddress = Globalization.Address.parseAddress(formattedAddressValue);
						
						if (parsedAddress) {
							localContact.addresses.push({
								streetAddress: parsedAddress.streetAddress,
								locality: parsedAddress.locality,
								region: parsedAddress.region,
								postalCode: parsedAddress.postalCode,
								country: parsedAddress.country,
								type: GoogleContactHandler._typeLookup(GoogleContactHandler._addressTypeMap, address.getRel(), false),
								primary: address.getPrimary()
							});
						} else {
							console.warn("Google contact sync: Globalization.Address.parseAddress returned falsy result: '" + parsedAddress + "'.  Address: '" + formattedAddressValue + "'");
						}
					}
				}
			}
		});
	}
};

GoogleContactHandler._copyUrlsToLocal = function (localContact, remoteContact) {
	var urls = remoteContact.getWebsites();
	localContact.urls = [];
	if (urls && Array.isArray(urls)) {
		urls.forEach(function (url) {
			localContact.urls.push({
				value: url.getHref(),
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._urlTypeMap, url.getRel(), false),
				primary: url.getPrimary()
			});
		});
	}
};

GoogleContactHandler._copyOrganizationsToLocal = function (localContact, remoteContact) {
	var organizations = remoteContact.getOrganizations();
	localContact.organizations = [];
	if (organizations && Array.isArray(organizations)) {
		organizations.forEach(function (organization) {
			var orgDept = organization.getOrgDepartment(),
				orgJobDesc = organization.getOrgJobDescription(),
				orgName = organization.getOrgName(),
				title = organization.getOrgTitle(),
				location = organization.getWhere();
			
			localContact.organizations.push({
				name: (orgName) ? orgName.getValue() : undefined,
				department: (orgDept) ? orgDept.getValue() : undefined,
				title: (title) ? title.getValue() : undefined,
				location: (location) ? Globalization.Address.parseAddress(location.getValueString()) : undefined,
				description: (orgJobDesc) ? orgJobDesc.getValue() : undefined,
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._organizationTypeMap, organization.getRel(), false),
				primary: organization.getPrimary()
			});
		});
	}
};

GoogleContactHandler._copyRelationsToLocal = function (localContact, remoteContact) {
	var relations = remoteContact.getRelations();
	localContact.relations = [];
	if (relations && Array.isArray(relations)) {
		relations.forEach(function (relation) {
			localContact.relations.push({
				value: relation.getValue(),
				type: GoogleContactHandler._typeLookup(GoogleContactHandler._relationTypeMap, relation.getRel(), false)
			});
		});
	}
};

GoogleContactHandler._copyPhotoToLocal = function (localContact, remoteContact) {
	var photoLinkObj = remoteContact.getContactPhotoLink();

	if (photoLinkObj) {
		this._copyPhotoLinkToLocal(localContact, remoteContact);
		localContact.photoEtag = photoLinkObj.gd$etag;
		localContact.photos = [];

		if (localContact.photoUrl && photoLinkObj.gd$etag) {
			localContact.photos.push({
				value: localContact.photoUrl,
				type: "type_square",
				etag: photoLinkObj.gd$etag
			});
		}
		
	}
};

GoogleContactHandler._copyPhotoLinkToLocal = function (localContact, remoteContact) {
	var photoLinkObj = remoteContact.getContactPhotoLink();
	if (photoLinkObj) {
		localContact.photoUrl = photoLinkObj.getHref();
	}
};

/*
 * Copy to remote methods
 */

GoogleContactHandler._copyNameToRemote = function (localContact, remoteContact) {
	var localNameObj = localContact.name,
		remoteNameObj,
		remotePrefixObj,
		remoteGivenNameObj,
		remoteMiddleNameObj,
		remoteFamilyNameObj,
		remoteSuffixObj,
		fullNameObj,
		namePartsArray,
		fullName = "";
	
	if (localNameObj) {
		remoteNameObj = remoteContact.getName() || new google.gdata.Name();
		
		if (localNameObj.honorificPrefix) {
			remotePrefixObj = remoteNameObj.getNamePrefix() || new google.gdata.NamePrefix();
			remotePrefixObj.setValue(localNameObj.honorificPrefix);
		}
		remoteNameObj.setNamePrefix(remotePrefixObj);
		
		if (localNameObj.givenName) {
			remoteGivenNameObj = remoteNameObj.getGivenName() || new google.gdata.GivenName();
			remoteGivenNameObj.setValue(localNameObj.givenName);
		}
		remoteNameObj.setGivenName(remoteGivenNameObj);
		
		if (localNameObj.middleName) {
			remoteMiddleNameObj = remoteNameObj.getAdditionalName() || new google.gdata.AdditionalName();
			remoteMiddleNameObj.setValue(localNameObj.middleName);
		}
		remoteNameObj.setAdditionalName(remoteMiddleNameObj);
		
		if (localNameObj.familyName) {
			remoteFamilyNameObj = remoteNameObj.getFamilyName() || new google.gdata.FamilyName();
			remoteFamilyNameObj.setValue(localNameObj.familyName);
		}
		remoteNameObj.setFamilyName(remoteFamilyNameObj);
		
		if (localNameObj.honorificSuffix) {
			remoteSuffixObj = remoteNameObj.getNameSuffix() || new google.gdata.NameSuffix();
			remoteSuffixObj.setValue(localNameObj.honorificSuffix);
		}
		remoteNameObj.setNameSuffix(remoteSuffixObj);
		
		//TODO: change this to use the globalization lib?
		namePartsArray = [localNameObj.honorificPrefix, localNameObj.givenName, localNameObj.middleName, localNameObj.familyName, localNameObj.honorificSuffix];
		fullName = namePartsArray.reduce(function (accumulator, curNamePart) {
			if (curNamePart) {
				if (accumulator) {
					accumulator += " ";
				}
				
				accumulator += curNamePart;
			}
			
			return accumulator;
		}, "");
		if (fullName) {
			fullNameObj = remoteNameObj.getFullName() || new google.gdata.FullName();
			fullNameObj.setValue(fullName);
		}
		remoteNameObj.setFullName(fullNameObj);
	}
	
	remoteContact.setName(remoteNameObj);
	remoteContact.setTitle({
		text: fullName
	});
};

GoogleContactHandler._copyNicknameToRemote = function (localContact, remoteContact) {
	var newRemoteNickname;
	if (localContact.nickname) {
		newRemoteNickname = new google.gdata.contacts.Nickname({
			value: localContact.nickname
		});
	}
	remoteContact.setNickname(newRemoteNickname);
};

GoogleContactHandler._copyGenderToRemote = function (localContact, remoteContact) {
	var newRemoteGender;
	if (localContact.gender) {
		newRemoteGender = new google.gdata.contacts.Gender({
			value: GoogleContactHandler._typeLookup(GoogleContactHandler._genderTypeMap, localContact.gender, true)
		});
	}
	remoteContact.setGender(newRemoteGender);
};

GoogleContactHandler._copyBirthdayToRemote = function (localContact, remoteContact) {
	var newRemoteBirthday;
	if (localContact.birthday) {
		newRemoteBirthday = new google.gdata.contacts.Birthday({
			when: localContact.birthday
		});
	}
	remoteContact.setBirthday(newRemoteBirthday);
};

GoogleContactHandler._copyAnniversaryToRemote = function (localContact, remoteContact) {
	var remoteEvents = remoteContact.getEvents(),
		curEventObj,
		whenObj,
		i,
		handledAnniversary = false,
		newWhenObj,
		newEventObj;
	
	if (remoteEvents && Array.isArray(remoteEvents)) {
		//grab the first one that claims to be an anniversary and store the date there.
		for (i = 0; i < remoteEvents.length; i += 1) {
			curEventObj = remoteEvents[i];
			if (curEventObj.getRel() === google.gdata.contacts.Event.REL_ANNIVERSARY) {
				if (localContact.anniversary) {
					//create/edit the current when object
					whenObj = curEventObj.getWhen() || new google.gdata.When();
					whenObj.setStartTime(google.gdata.DateTime.fromIso8601(localContact.anniversary));
					
					//store the when in the current event object and the event back into the array
					curEventObj.setWhen(whenObj);
					remoteEvents[i] = curEventObj;
				} else {
					//remove the current element, since the user deleted the anniversary
					remoteEvents.splice(i, 1);
				}
				
				handledAnniversary = true;
				break;
			}
		}
	}
	
	//but if we didn't store it above for some reason, but it exists locally, let's create a new event and add it to that
	if (!handledAnniversary && localContact.anniversary) {
		newWhenObj = new google.gdata.When();
		newWhenObj.setStartTime(google.gdata.DateTime.fromIso8601(localContact.anniversary));
		
		newEventObj = new google.gdata.contacts.Event();
		newEventObj.setWhen(newWhenObj);
		newEventObj.setRel(google.gdata.contacts.Event.REL_ANNIVERSARY);
		
		remoteEvents.push(newEventObj);
	}
	
	remoteContact.setEvents(remoteEvents);
};

GoogleContactHandler._copyNoteToRemote = function (localContact, remoteContact) {
	var newRemoteNote;
	if (localContact.note) {
		newRemoteNote = new google.gdata.atom.Text({
			text: localContact.note
		});
	}
	remoteContact.setContent(newRemoteNote);
};

GoogleContactHandler._copyCustomFieldsToRemote = function (localContact, remoteContact) {
	var newRemoteCustomFields = [];
	
	if (localContact.customFields && Array.isArray(localContact.customFields)) {
		localContact.customFields.forEach(function (localCustomField) {
			var newCustomField = new google.gdata.contacts.UserDefinedField({
				value: localCustomField.value
			});
			//this should use customField.setKey() or just pass the key in the object passed to the 
			//constructor above, but that doesn't seem to set the right property
			newCustomField.key = localCustomField.name;
			newRemoteCustomFields.push(newCustomField);
		});
	}
	
	remoteContact.setUserDefinedFields(newRemoteCustomFields);
};

GoogleContactHandler._copyPhoneNumbersToRemote = function (localContact, remoteContact) {
	var newRemotePhoneNumbers = [],
		oldRemotePhoneNumbers = remoteContact.getPhoneNumbers();
	
	if (localContact.phoneNumbers && Array.isArray(localContact.phoneNumbers)) {
		localContact.phoneNumbers.forEach(function (localPhoneNumber) {
			//look for this phone number from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			var remoteCopy = _.detect(oldRemotePhoneNumbers, function (remotePhoneNumber) {
				return (localPhoneNumber.value === remotePhoneNumber.getValue());
			});
			
			if (remoteCopy) {
				remoteCopy.setValue(localPhoneNumber.value);
				remoteCopy.setPrimary(localPhoneNumber.primary);
				remoteCopy.setLabel(undefined); //clear the label so that we can set the rel - can only have one
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._phoneTypeMap, localPhoneNumber.type, true));
				
				newRemotePhoneNumbers.push(remoteCopy);
			} else {
				newRemotePhoneNumbers.push(new google.gdata.PhoneNumber({
					value: localPhoneNumber.value,
					primary: localPhoneNumber.primary,
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._phoneTypeMap, localPhoneNumber.type, true)
				}));
			}
		});
	}
	
	remoteContact.setPhoneNumbers(newRemotePhoneNumbers);
};

GoogleContactHandler._copyEmailsToRemote = function (localContact, remoteContact) {
	var newRemoteEmails = [],
		oldRemoteEmails = remoteContact.getEmailAddresses();
	
	if (localContact.emails && Array.isArray(localContact.emails)) {
		localContact.emails.forEach(function (localEmail) {
			//look for this email from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			var remoteCopy = _.detect(oldRemoteEmails, function (remoteEmail) {
				return (localEmail.value === remoteEmail.getAddress());
			});
			
			if (remoteCopy) {
				remoteCopy.setAddress(localEmail.value);
				remoteCopy.setPrimary(localEmail.primary);
				remoteCopy.setLabel(undefined); //clear the label so that we can set the rel - can only have one
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._emailTypeMap, localEmail.type, true));
				
				newRemoteEmails.push(remoteCopy);
			} else {
				newRemoteEmails.push(new google.gdata.Email({
					address: localEmail.value,
					primary: localEmail.primary,
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._emailTypeMap, localEmail.type, true)
				}));
			}
		});
	}
	
	remoteContact.setEmailAddresses(newRemoteEmails);
};

GoogleContactHandler._copyImsToRemote = function (localContact, remoteContact) {
	var newRemoteIms = [],
		oldRemoteIms = remoteContact.getImAddresses();
	
	if (localContact.ims && Array.isArray(localContact.ims)) {
		localContact.ims.forEach(function (localIm) {
			//look for this im from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			var remoteCopy = _.detect(oldRemoteIms, function (remoteIm) {
				return (localIm.value === remoteIm.getAddress());
			});
			
			if (remoteCopy) {
				remoteCopy.setAddress(localIm.value);
				remoteCopy.setPrimary(localIm.primary);
				remoteCopy.setProtocol(GoogleContactHandler._typeLookup(GoogleContactHandler._imTypeMap, localIm.type, true));
				remoteCopy.setLabel(undefined); //clear the label so that we can set the rel - can only have one
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._imLabelMap, localIm.label, true));
				
				newRemoteIms.push(remoteCopy);
			} else {
				newRemoteIms.push(new google.gdata.Im({
					address: localIm.value,
					primary: localIm.primary,
					protocol: GoogleContactHandler._typeLookup(GoogleContactHandler._imTypeMap, localIm.type, true),
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._imLabelMap, localIm.label, true)
				}));
			}
		});
	}
	
	remoteContact.setImAddresses(newRemoteIms);
};

GoogleContactHandler._copyAddressesToRemote = function (localContact, remoteContact) {
	var newRemoteAddresses = [],
		oldRemoteAddresses = remoteContact.getStructuredPostalAddresses();
	
	if (localContact.addresses && Array.isArray(localContact.addresses)) {
		localContact.addresses.forEach(function (localAddress) {
			var localFreeformAddress = Globalization.Address.formatAddress(localAddress),
				remoteCopy,
				streetAddressObj,
				localityObj,
				regionObj,
				postalCodeObj,
				countryObj,
				formattedAddressObj;
			
			//look for this address from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			remoteCopy = _.detect(oldRemoteAddresses, function (remoteAddress) {
				var remoteStreetAddress = remoteAddress.getStreet() && remoteAddress.getStreet().getValue(),
					remoteLocality = remoteAddress.getCity() && remoteAddress.getCity().getValue(),
					remoteRegion = remoteAddress.getRegion() && remoteAddress.getRegion().getValue(),
					remotePostalCode = remoteAddress.getPostcode() && remoteAddress.getPostcode().getValue(),
					remoteCountry = remoteAddress.getCountry() && remoteAddress.getCountry().getValue(),
					remoteFreeformAddress = remoteAddress.getFormattedAddress() && remoteAddress.getFormattedAddress().getValue();
				
				//if the remote copy has any of the structured fields, we compare using those
				if (remoteStreetAddress || remoteLocality || remoteRegion || remotePostalCode || remoteCountry) {
					return remoteStreetAddress === localAddress.streetAddress && 
							remoteLocality === localAddress.locality && 
							remoteRegion === localAddress.region && 
							remotePostalCode === localAddress.postalCode && 
							remoteCountry === localAddress.country && 
							remoteAddress.getRel() === localAddress.type;
				} else {
					//else we use the freeform field
					return remoteFreeformAddress === localFreeformAddress && 
							remoteAddress.getRel() === localAddress.type;
				}
			});
			
			if (remoteCopy) {
				if (localAddress.streetAddress) {
					streetAddressObj = remoteCopy.getStreet() || new google.gdata.Street();
					streetAddressObj.setValue(localAddress.streetAddress);
				}
				remoteCopy.setStreet(streetAddressObj);
				
				if (localAddress.locality) {
					localityObj = remoteCopy.getCity() || new google.gdata.City();
					localityObj.setValue(localAddress.locality);
				}
				remoteCopy.setCity(localityObj);
				
				if (localAddress.region) {
					regionObj = remoteCopy.getRegion() || new google.gdata.Region();
					regionObj.setValue(localAddress.region);
				}
				remoteCopy.setRegion(regionObj);
				
				if (localAddress.postalCode) {
					postalCodeObj = remoteCopy.getPostcode() || new google.gdata.PostCode();
					postalCodeObj.setValue(localAddress.postalCode);
				}
				remoteCopy.setPostcode(postalCodeObj);
				
				if (localAddress.country) {
					countryObj = remoteCopy.getCountry() || new google.gdata.Country();
					countryObj.setValue(localAddress.country);
				}
				remoteCopy.setCountry(countryObj);
				
				formattedAddressObj = remoteCopy.getFormattedAddress() || new google.gdata.FormattedAddress();
				formattedAddressObj.setValue(localFreeformAddress);
				remoteCopy.setFormattedAddress(formattedAddressObj);
				
				remoteCopy.setLabel(undefined); //clear the label so that we can set the rel - can only have one
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._addressTypeMap, localAddress.type, true));
				remoteCopy.setPrimary(localAddress.primary);
				
				newRemoteAddresses.push(remoteCopy);
			} else {
				if (localAddress.streetAddress) {
					streetAddressObj = new google.gdata.Street({
						value: localAddress.streetAddress
					});
				}
				if (localAddress.locality) {
					localityObj = new google.gdata.City({
						value: localAddress.locality
					});
				}
				if (localAddress.region) {
					regionObj = new google.gdata.Region({
						value: localAddress.region
					});
				}
				if (localAddress.country) {
					countryObj = new google.gdata.PostCode({
						value: localAddress.country
					});
				}
				if (localAddress.postalCode) {
					postalCodeObj = new google.gdata.Country({
						value: localAddress.postalCode
					});
				}
				formattedAddressObj = new google.gdata.FormattedAddress({
					value: localFreeformAddress
				});
				
				newRemoteAddresses.push(new google.gdata.StructuredPostalAddress({
					street: streetAddressObj,
					city: localityObj,
					region: regionObj,
					postcode: postalCodeObj,
					country: countryObj,
					formattedAddress: formattedAddressObj,
					primary: localAddress.primary,
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._addressTypeMap, localAddress.type, true)
				}));
			}
		});
	}
	
	remoteContact.setStructuredPostalAddresses(newRemoteAddresses);
};

GoogleContactHandler._copyUrlsToRemote = function (localContact, remoteContact) {
	var newRemoteUrls = [],
		oldRemoteUrls = remoteContact.getWebsites();
	
	if (localContact.urls && Array.isArray(localContact.urls)) {
		localContact.urls.forEach(function (localUrl) {
			//look for this url from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			var remoteCopy = _.detect(oldRemoteUrls, function (remoteUrl) {
				return (localUrl.value === remoteUrl.getHref());
			});
			
			if (remoteCopy) {
				remoteCopy.setHref(localUrl.value);
				remoteCopy.setPrimary(localUrl.primary);
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._urlTypeMap, localUrl.type, true));
				
				newRemoteUrls.push(remoteCopy);
			} else {
				newRemoteUrls.push(new google.gdata.contacts.Website({
					href: localUrl.value,
					primary: localUrl.primary,
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._urlTypeMap, localUrl.type, true)
				}));
			}
		});
	}
	
	remoteContact.setWebsites(newRemoteUrls);
};

GoogleContactHandler._copyOrganizationsToRemote = function (localContact, remoteContact) {
	var newRemoteOrganizations = [],
		oldRemoteOrganizations = remoteContact.getOrganizations();
	
	if (localContact.organizations && Array.isArray(localContact.organizations)) {
		localContact.organizations.forEach(function (localOrganization) {
			var remoteCopy,
				orgNameObj,
				orgDeptObj,
				orgTitleObj,
				orgDescObj,
				whereObj,
				locationString;
			
			//look for this organization from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			remoteCopy = _.detect(oldRemoteOrganizations, function (remoteOrganization) {
				var remoteOrgName = remoteOrganization.getOrgName(),
					remoteTitle = remoteOrganization.getOrgTitle();
				
				//we're using company name, title, and type as the distinguishers
				return (remoteOrgName && remoteOrgName.getValue() === localOrganization.name &&
						remoteTitle && remoteTitle.getValue() === localOrganization.title && 
						remoteOrganization.getRel() === localOrganization.type);
			});
			
			if (remoteCopy) {
				if (localOrganization.name) {
					orgNameObj = remoteCopy.getOrgName() || new google.gdata.OrgName();
					orgNameObj.setValue(localOrganization.name);
				}
				remoteCopy.setOrgName(orgNameObj);
				
				if (localOrganization.department) {
					orgDeptObj = remoteCopy.getOrgDepartment() || new google.gdata.OrgDepartment();
					orgDeptObj.setValue(localOrganization.department);
				}
				remoteCopy.setOrgDepartment(orgDeptObj);
				
				if (localOrganization.title) {
					orgTitleObj = remoteCopy.getOrgTitle() || new google.gdata.OrgTitle();
					orgTitleObj.setValue(localOrganization.title);
				}
				remoteCopy.setOrgTitle(orgTitleObj);
				
				if (localOrganization.description) {
					orgDescObj = remoteCopy.getOrgJobDescription() || new google.gdata.OrgJobDescription();
					orgDescObj.setValue(localOrganization.description);
				}
				remoteCopy.setOrgJobDescription(orgDescObj);
				
				if (localOrganization.location) {
					whereObj = remoteCopy.getWhere() || new google.gdata.Where();
					whereObj.setValueString(Globalization.Address.formatAddress(localOrganization.location));
				}
				remoteCopy.setWhere(whereObj);
				
				remoteCopy.setLabel(undefined); //clear the label so that we can set the rel - can only have one
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._organizationTypeMap, localOrganization.type, true));
				remoteCopy.setPrimary(localOrganization.primary);
				
				newRemoteOrganizations.push(remoteCopy);
			} else {
				locationString = Globalization.Address.formatAddress(localOrganization.location);
				if (localOrganization.name || localOrganization.department || localOrganization.title || localOrganization.description || locationString) {
					if (localOrganization.name) {
						orgNameObj = new google.gdata.OrgName({
							value: localOrganization.name
						});
					}
					if (localOrganization.department) {
						orgDeptObj = new google.gdata.OrgDepartment({
							value: localOrganization.department
						});
					}
					if (localOrganization.title) {
						orgTitleObj = new google.gdata.OrgTitle({
							value: localOrganization.title
						});
					}
					if (localOrganization.description) {
						orgDescObj = new google.gdata.OrgJobDescription({
							value: localOrganization.description
						});
					}
					if (locationString) {
						whereObj = new google.gdata.Where({
							valueString: locationString
						});
					}
					newRemoteOrganizations.push(new google.gdata.Organization({
						orgName: orgNameObj,
						orgDepartment: orgDeptObj,
						orgTitle: orgTitleObj,
						orgJobDescription: orgDescObj,
						where: whereObj,
						primary: localOrganization.primary,
						rel: GoogleContactHandler._typeLookup(GoogleContactHandler._organizationTypeMap, localOrganization.type, true)
					}));
				}
			}
		});
	}
	
	remoteContact.setOrganizations(newRemoteOrganizations);
};

GoogleContactHandler._copyRelationsToRemote = function (localContact, remoteContact) {
	var newRemoteRelations = [],
		oldRemoteRelations = remoteContact.getRelations();
	
	if (localContact.relations && Array.isArray(localContact.relations)) {
		localContact.relations.forEach(function (localRelation) {
			//look for this relation from the server.  if it exists we use it, so we preserve extra fields that we don't store.
			var remoteCopy = _.detect(oldRemoteRelations, function (remoteRelation) {
				return (localRelation.value === remoteRelation.getValue() && 
						localRelation.type === remoteRelation.getRel());
			});
			
			if (remoteCopy) {
				remoteCopy.setValue(localRelation.value);
				remoteCopy.setRel(GoogleContactHandler._typeLookup(GoogleContactHandler._relationTypeMap, localRelation.type, true));
				
				newRemoteRelations.push(remoteCopy);
			} else {
				newRemoteRelations.push(new google.gdata.contacts.Relation({
					value: localRelation.value,
					rel: GoogleContactHandler._typeLookup(GoogleContactHandler._relationTypeMap, localRelation.type, true)
				}));
			}
		});
	}
	
	remoteContact.setRelations(newRemoteRelations);
};

GoogleContactHandler._copyPhotoToRemote = function (localContact, remoteContact) {
	var photoLinkObj,
	photoObj;
	try {
		photoLinkObj = remoteContact.getContactPhotoLink();
	} catch (e) {
		console.warn("Google remote contact photo link could not be read");
		return;
	}
	if (photoLinkObj && localContact.photoUrl) {
		photoLinkObj.setHref(localContact.photoUrl);
	} else if (!photoLinkObj && localContact.photoUrl) {
		photoObj = new google.gdata.atom.Link();
		photoObj.setHref(localContact.photoUrl);
		photoObj.setRel("http://schemas.google.com/contacts/2008/rel#photo");
		photoObj.setType("image/*");
		remoteContact.link.push(photoObj);
	}
};
